/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.client;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.WeightedRandomSelection;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.proto.BookieClient;
import org.apache.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BookieInfoReader {
    private static final Logger LOG = LoggerFactory.getLogger(BookieInfoReader.class);
    private static final long GET_BOOKIE_INFO_REQUEST_FLAGS = 3L;
    private final ScheduledExecutorService scheduler;
    private final BookKeeper bk;
    private final ClientConfiguration conf;
    private final BookieInfoMap bookieInfoMap = new BookieInfoMap();
    private final InstanceState instanceState = new InstanceState();
    private int totalSent = 0;
    private int completedCnt = 0;
    private int errorCnt = 0;

    BookieInfoReader(BookKeeper bk, ClientConfiguration conf, ScheduledExecutorService scheduler) {
        this.bk = bk;
        this.conf = conf;
        this.scheduler = scheduler;
    }

    public void start() {
        this.scheduler.scheduleAtFixedRate(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                BookieInfoReader bookieInfoReader = BookieInfoReader.this;
                synchronized (bookieInfoReader) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Running periodic BookieInfo scan");
                    }
                    try {
                        Collection<BookieSocketAddress> updatedBookies = ((BookieInfoReader)BookieInfoReader.this).bk.bookieWatcher.getBookies();
                        BookieInfoReader.this.bookieInfoMap.updateBookies(updatedBookies);
                    }
                    catch (BKException e) {
                        LOG.info("Got exception while querying bookies from watcher, rerunning after {}s", (Object)BookieInfoReader.this.conf.getGetBookieInfoRetryIntervalSeconds(), (Object)e);
                        BookieInfoReader.this.scheduler.schedule(this, (long)BookieInfoReader.this.conf.getGetBookieInfoRetryIntervalSeconds(), TimeUnit.SECONDS);
                        return;
                    }
                    if (BookieInfoReader.this.instanceState.tryStartFull()) {
                        BookieInfoReader.this.getReadWriteBookieInfo();
                    }
                }
            }
        }, 0L, this.conf.getGetBookieInfoIntervalSeconds(), TimeUnit.SECONDS);
    }

    private void submitTask() {
        this.scheduler.submit(() -> this.getReadWriteBookieInfo());
    }

    private void submitTaskWithDelay(int delaySeconds) {
        this.scheduler.schedule(() -> this.getReadWriteBookieInfo(), (long)delaySeconds, TimeUnit.SECONDS);
    }

    synchronized void availableBookiesChanged(Set<BookieSocketAddress> updatedBookiesList) {
        if (LOG.isInfoEnabled()) {
            LOG.info("Scheduling bookie info read due to changes in available bookies.");
        }
        this.bookieInfoMap.updateBookies(updatedBookiesList);
        if (this.instanceState.tryStartPartial()) {
            this.submitTask();
        }
    }

    synchronized Optional<Long> getFreeDiskSpace(BookieSocketAddress bookie) {
        BookieInfo bookieInfo = this.bookieInfoMap.getInfo(bookie);
        if (bookieInfo != null) {
            return Optional.of(bookieInfo.getFreeDiskSpace());
        }
        return Optional.empty();
    }

    synchronized void getReadWriteBookieInfo() {
        Collection<BookieSocketAddress> toScan;
        State queuedType = this.instanceState.getAndClearQueuedType();
        if (queuedType == State.FULL) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Doing full scan");
            }
            toScan = this.bookieInfoMap.getFullScanTargets();
        } else if (queuedType == State.PARTIAL) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Doing partial scan");
            }
            toScan = this.bookieInfoMap.getPartialScanTargets();
        } else {
            if (LOG.isErrorEnabled()) {
                LOG.error("Invalid state, queuedType cannot be UNQUEUED in getReadWriteBookieInfo");
            }
            assert (queuedType != State.UNQUEUED);
            return;
        }
        BookieClient bkc = this.bk.getBookieClient();
        long requested = 3L;
        this.totalSent = 0;
        this.completedCnt = 0;
        this.errorCnt = 0;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Getting bookie info for: {}", toScan);
        }
        for (BookieSocketAddress b : toScan) {
            bkc.getBookieInfo(b, 3L, new BookkeeperInternalCallbacks.GetBookieInfoCallback(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                void processReadInfoComplete(int rc, BookieInfo bInfo, Object ctx) {
                    BookieInfoReader bookieInfoReader = BookieInfoReader.this;
                    synchronized (bookieInfoReader) {
                        BookieSocketAddress b = (BookieSocketAddress)ctx;
                        if (rc != 0) {
                            if (LOG.isErrorEnabled()) {
                                LOG.error("Reading bookie info from bookie {} failed due to error: {}.", (Object)b, (Object)rc);
                            }
                            BookieInfoReader.this.bookieInfoMap.clearInfo(b);
                            BookieInfoReader.this.errorCnt++;
                        } else {
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("Bookie Info for bookie {} is {}", (Object)b, (Object)bInfo);
                            }
                            BookieInfoReader.this.bookieInfoMap.gotInfo(b, bInfo);
                        }
                        BookieInfoReader.this.completedCnt++;
                        if (BookieInfoReader.this.totalSent == BookieInfoReader.this.completedCnt) {
                            BookieInfoReader.this.onExit();
                        }
                    }
                }

                @Override
                public void getBookieInfoComplete(final int rc, final BookieInfo bInfo, final Object ctx) {
                    BookieInfoReader.this.scheduler.submit(new Runnable(){

                        @Override
                        public void run() {
                            this.processReadInfoComplete(rc, bInfo, ctx);
                        }
                    });
                }
            }, b);
            ++this.totalSent;
        }
        if (this.totalSent == 0) {
            this.onExit();
        }
    }

    void onExit() {
        this.bk.placementPolicy.updateBookieInfo(this.bookieInfoMap.getBookieMap());
        if (this.errorCnt > 0) {
            if (LOG.isInfoEnabled()) {
                LOG.info("Rescheduling in {}s due to errors", (Object)this.conf.getGetBookieInfoIntervalSeconds());
            }
            this.instanceState.tryStartPartial();
            this.submitTaskWithDelay(this.conf.getGetBookieInfoRetryIntervalSeconds());
        } else if (this.instanceState.completeUnlessQueued()) {
            if (LOG.isInfoEnabled()) {
                LOG.info("Rescheduling, another scan is pending");
            }
            this.submitTask();
        }
    }

    Map<BookieSocketAddress, BookieInfo> getBookieInfo() throws BKException, InterruptedException {
        BookieClient bkc = this.bk.getBookieClient();
        final AtomicInteger totalSent = new AtomicInteger();
        final AtomicInteger totalCompleted = new AtomicInteger();
        final ConcurrentHashMap<BookieSocketAddress, BookieInfo> map = new ConcurrentHashMap<BookieSocketAddress, BookieInfo>();
        final CountDownLatch latch = new CountDownLatch(1);
        long requested = 3L;
        Collection<BookieSocketAddress> bookies = this.bk.bookieWatcher.getBookies();
        bookies.addAll(this.bk.bookieWatcher.getReadOnlyBookies());
        totalSent.set(bookies.size());
        for (BookieSocketAddress b : bookies) {
            bkc.getBookieInfo(b, requested, new BookkeeperInternalCallbacks.GetBookieInfoCallback(){

                @Override
                public void getBookieInfoComplete(int rc, BookieInfo bInfo, Object ctx) {
                    BookieSocketAddress b = (BookieSocketAddress)ctx;
                    if (rc != 0) {
                        if (LOG.isErrorEnabled()) {
                            LOG.error("Reading bookie info from bookie {} failed due to error: {}.", (Object)b, (Object)rc);
                        }
                    } else {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Free disk space on bookie {} is {}.", (Object)b, (Object)bInfo.getFreeDiskSpace());
                        }
                        map.put(b, bInfo);
                    }
                    if (totalCompleted.incrementAndGet() == totalSent.get()) {
                        latch.countDown();
                    }
                }
            }, b);
        }
        try {
            latch.await();
        }
        catch (InterruptedException e) {
            LOG.error("Received InterruptedException ", (Throwable)e);
            throw e;
        }
        return map;
    }

    private static class InstanceState {
        private boolean running = false;
        private State queuedType = State.UNQUEUED;

        private InstanceState() {
        }

        private boolean shouldStart() {
            if (this.running) {
                return false;
            }
            this.running = true;
            return true;
        }

        public boolean tryStartFull() {
            this.queuedType = State.FULL;
            return this.shouldStart();
        }

        public boolean tryStartPartial() {
            if (this.queuedType == State.UNQUEUED) {
                this.queuedType = State.PARTIAL;
            }
            return this.shouldStart();
        }

        public State getAndClearQueuedType() {
            State ret = this.queuedType;
            this.queuedType = State.UNQUEUED;
            return ret;
        }

        public boolean completeUnlessQueued() {
            if (this.queuedType == State.UNQUEUED) {
                this.running = false;
                return false;
            }
            return true;
        }
    }

    public static enum State {
        UNQUEUED,
        PARTIAL,
        FULL;

    }

    private static class BookieInfoMap {
        private final Map<BookieSocketAddress, BookieInfo> infoMap = new HashMap<BookieSocketAddress, BookieInfo>();
        private Collection<BookieSocketAddress> mostRecentlyReportedBookies = new ArrayList<BookieSocketAddress>();

        private BookieInfoMap() {
        }

        public void updateBookies(Collection<BookieSocketAddress> updatedBookieSet) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("updateBookies: current: {}, new: {}", this.mostRecentlyReportedBookies, updatedBookieSet);
            }
            this.infoMap.keySet().retainAll(updatedBookieSet);
            this.mostRecentlyReportedBookies = updatedBookieSet;
        }

        public Collection<BookieSocketAddress> getPartialScanTargets() {
            return CollectionUtils.subtract(this.mostRecentlyReportedBookies, this.infoMap.keySet());
        }

        public Collection<BookieSocketAddress> getFullScanTargets() {
            return this.mostRecentlyReportedBookies;
        }

        public BookieInfo getInfo(BookieSocketAddress bookie) {
            return this.infoMap.get(bookie);
        }

        public void clearInfo(BookieSocketAddress bookie) {
            this.infoMap.remove(bookie);
        }

        public void gotInfo(BookieSocketAddress bookie, BookieInfo info) {
            this.infoMap.put(bookie, info);
        }

        public Map<BookieSocketAddress, BookieInfo> getBookieMap() {
            return this.infoMap;
        }
    }

    public static class BookieInfo
    implements WeightedRandomSelection.WeightedObject {
        private final long freeDiskSpace;
        private final long totalDiskSpace;

        public BookieInfo() {
            this(0L, 0L);
        }

        public BookieInfo(long totalDiskSpace, long freeDiskSpace) {
            this.totalDiskSpace = totalDiskSpace;
            this.freeDiskSpace = freeDiskSpace;
        }

        public long getFreeDiskSpace() {
            return this.freeDiskSpace;
        }

        public long getTotalDiskSpace() {
            return this.totalDiskSpace;
        }

        @Override
        public long getWeight() {
            return this.freeDiskSpace;
        }

        public String toString() {
            return "FreeDiskSpace: " + this.freeDiskSpace + " TotalDiskCapacity: " + this.totalDiskSpace;
        }
    }
}

