/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.replication;

import bk-shade.com.google.common.base.Stopwatch;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.SortedMap;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.bookkeeper.bookie.BookieThread;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.BookKeeperAdmin;
import org.apache.bookkeeper.client.LedgerChecker;
import org.apache.bookkeeper.client.LedgerFragment;
import org.apache.bookkeeper.client.LedgerHandle;
import org.apache.bookkeeper.client.LedgerMetadata;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManagerFactory;
import org.apache.bookkeeper.meta.LedgerUnderreplicationManager;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.bookkeeper.replication.ReplicationException;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplicationWorker
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicationWorker.class);
    private final LedgerUnderreplicationManager underreplicationManager;
    private final ServerConfiguration conf;
    private final ZooKeeper zkc;
    private volatile boolean workerRunning = false;
    private volatile boolean isInReadOnlyMode = false;
    private final BookKeeperAdmin admin;
    private final LedgerChecker ledgerChecker;
    private final BookieSocketAddress targetBookie;
    private final BookKeeper bkc;
    private final Thread workerThread;
    private final long openLedgerRereplicationGracePeriod;
    private final Timer pendingReplicationTimer;
    private final OpStatsLogger rereplicateOpStats;
    private final Counter numLedgersReplicated;

    public ReplicationWorker(ZooKeeper zkc, ServerConfiguration conf, BookieSocketAddress targetBKAddr) throws ReplicationException.CompatibilityException, KeeperException, InterruptedException, IOException {
        this(zkc, conf, targetBKAddr, (StatsLogger)NullStatsLogger.INSTANCE);
    }

    public ReplicationWorker(ZooKeeper zkc, ServerConfiguration conf, BookieSocketAddress targetBKAddr, StatsLogger statsLogger) throws ReplicationException.CompatibilityException, KeeperException, InterruptedException, IOException {
        this.zkc = zkc;
        this.conf = conf;
        this.targetBookie = targetBKAddr;
        LedgerManagerFactory mFactory = LedgerManagerFactory.newLedgerManagerFactory(this.conf, this.zkc);
        this.underreplicationManager = mFactory.newLedgerUnderreplicationManager();
        this.bkc = BookKeeper.forConfig(new ClientConfiguration(conf)).zk(zkc).statsLogger(statsLogger.scope("bk_client")).build();
        this.admin = new BookKeeperAdmin(this.bkc, statsLogger);
        this.ledgerChecker = new LedgerChecker(this.bkc);
        this.workerThread = new BookieThread(this, "ReplicationWorker");
        this.openLedgerRereplicationGracePeriod = conf.getOpenLedgerRereplicationGracePeriod();
        this.pendingReplicationTimer = new Timer("PendingReplicationTimer");
        this.rereplicateOpStats = statsLogger.getOpStatsLogger("rereplicate");
        this.numLedgersReplicated = statsLogger.getCounter("NUM_FULL_OR_PARTIAL_LEDGERS_REPLICATED");
    }

    public void start() {
        this.workerThread.start();
    }

    @Override
    public void run() {
        this.workerRunning = true;
        while (this.workerRunning) {
            try {
                this.rereplicate();
            }
            catch (InterruptedException e) {
                LOG.info("InterruptedException while replicating fragments", (Throwable)e);
                this.shutdown();
                Thread.currentThread().interrupt();
                return;
            }
            catch (BKException e) {
                LOG.error("BKException while replicating fragments", (Throwable)e);
                if (e instanceof BKException.BKWriteOnReadOnlyBookieException) {
                    this.waitTillTargetBookieIsWritable();
                    continue;
                }
                ReplicationWorker.waitBackOffTime();
            }
            catch (ReplicationException.UnavailableException e) {
                LOG.error("UnavailableException while replicating fragments", (Throwable)e);
                ReplicationWorker.waitBackOffTime();
            }
        }
        LOG.info("ReplicationWorker exited loop!");
    }

    private static void waitBackOffTime() {
        try {
            Thread.sleep(5000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private void waitTillTargetBookieIsWritable() {
        LOG.info("Waiting for target bookie {} to be back in read/write mode", (Object)this.targetBookie);
        while (this.workerRunning && this.admin.getReadOnlyBookies().contains(this.targetBookie)) {
            this.isInReadOnlyMode = true;
            ReplicationWorker.waitBackOffTime();
        }
        this.isInReadOnlyMode = false;
        LOG.info("Target bookie {} is back in read/write mode", (Object)this.targetBookie);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void rereplicate() throws InterruptedException, BKException, ReplicationException.UnavailableException {
        long ledgerIdToReplicate = this.underreplicationManager.getLedgerToRereplicate();
        Stopwatch stopwatch = Stopwatch.createStarted();
        boolean success = false;
        try {
            success = this.rereplicate(ledgerIdToReplicate);
        }
        finally {
            long latencyMillis = stopwatch.stop().elapsed(TimeUnit.MILLISECONDS);
            if (success) {
                this.rereplicateOpStats.registerSuccessfulEvent(latencyMillis, TimeUnit.MILLISECONDS);
            } else {
                this.rereplicateOpStats.registerFailedEvent(latencyMillis, TimeUnit.MILLISECONDS);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean rereplicate(long ledgerIdToReplicate) throws InterruptedException, BKException, ReplicationException.UnavailableException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Going to replicate the fragments of the ledger: {}", (Object)ledgerIdToReplicate);
        }
        try (LedgerHandle lh = this.admin.openLedgerNoRecovery(ledgerIdToReplicate);){
            Set<LedgerFragment> fragments = this.getUnderreplicatedFragments(lh);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Founds fragments {} for replication from ledger: {}", fragments, (Object)ledgerIdToReplicate);
            }
            boolean foundOpenFragments = false;
            long numFragsReplicated = 0L;
            for (LedgerFragment ledgerFragment : fragments) {
                if (!ledgerFragment.isClosed()) {
                    foundOpenFragments = true;
                    continue;
                }
                if (this.isTargetBookieExistsInFragmentEnsemble(lh, ledgerFragment)) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("Target Bookie[{}] found in the fragment ensemble: {}", (Object)this.targetBookie, ledgerFragment.getEnsemble());
                    continue;
                }
                try {
                    this.admin.replicateLedgerFragment(lh, ledgerFragment, this.targetBookie);
                    ++numFragsReplicated;
                }
                catch (BKException.BKBookieHandleNotAvailableException e) {
                    LOG.warn("BKBookieHandleNotAvailableException while replicating the fragment", (Throwable)e);
                }
                catch (BKException.BKLedgerRecoveryException e) {
                    LOG.warn("BKLedgerRecoveryException while replicating the fragment", (Throwable)e);
                    if (!this.admin.getReadOnlyBookies().contains(this.targetBookie)) continue;
                    this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
                    throw new BKException.BKWriteOnReadOnlyBookieException();
                }
            }
            if (numFragsReplicated > 0L) {
                this.numLedgersReplicated.inc();
            }
            if (foundOpenFragments || this.isLastSegmentOpenAndMissingBookies(lh)) {
                this.deferLedgerLockRelease(ledgerIdToReplicate);
                boolean bl = false;
                return bl;
            }
            fragments = this.getUnderreplicatedFragments(lh);
            if (fragments.size() == 0) {
                LOG.info("Ledger replicated successfully. ledger id is: " + ledgerIdToReplicate);
                this.underreplicationManager.markLedgerReplicated(ledgerIdToReplicate);
                boolean bl = true;
                return bl;
            }
            this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
            boolean bl = false;
            return bl;
        }
        catch (BKException.BKNoSuchLedgerExistsException e) {
            LOG.info("BKNoSuchLedgerExistsException while opening ledger for replication. Other clients might have deleted the ledger. So, no harm to continue");
            this.underreplicationManager.markLedgerReplicated(ledgerIdToReplicate);
            return false;
        }
        catch (BKException.BKReadException e) {
            LOG.info("BKReadException while opening ledger for replication. Enough Bookies might not have availableSo, no harm to continue");
            this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
            return false;
        }
        catch (BKException.BKBookieHandleNotAvailableException e) {
            LOG.info("BKBookieHandleNotAvailableException while opening ledger for replication. Enough Bookies might not have availableSo, no harm to continue");
            this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
            return false;
        }
    }

    private boolean isLastSegmentOpenAndMissingBookies(LedgerHandle lh) throws BKException {
        LedgerMetadata md = this.admin.getLedgerMetadata(lh);
        if (md.isClosed()) {
            return false;
        }
        SortedMap<Long, ArrayList<BookieSocketAddress>> ensembles = this.admin.getLedgerMetadata(lh).getEnsembles();
        ArrayList finalEnsemble = (ArrayList)ensembles.get(ensembles.lastKey());
        Collection<BookieSocketAddress> available = this.admin.getAvailableBookies();
        for (BookieSocketAddress b : finalEnsemble) {
            if (available.contains(b)) continue;
            return true;
        }
        return false;
    }

    private Set<LedgerFragment> getUnderreplicatedFragments(LedgerHandle lh) throws InterruptedException {
        CheckerCallback checkerCb = new CheckerCallback();
        this.ledgerChecker.checkLedger(lh, checkerCb);
        Set<LedgerFragment> fragments = checkerCb.waitAndGetResult();
        return fragments;
    }

    private void deferLedgerLockRelease(final long ledgerId) {
        long gracePeriod = this.openLedgerRereplicationGracePeriod;
        TimerTask timerTask = new TimerTask(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             * Enabled aggressive block sorting
             * Enabled unnecessary exception pruning
             * Enabled aggressive exception aggregation
             */
            @Override
            public void run() {
                LedgerHandle lh = null;
                try {
                    LedgerFragment fragment;
                    lh = ReplicationWorker.this.admin.openLedgerNoRecovery(ledgerId);
                    if (ReplicationWorker.this.isLastSegmentOpenAndMissingBookies(lh)) {
                        lh = ReplicationWorker.this.admin.openLedger(ledgerId);
                    }
                    Set fragments = ReplicationWorker.this.getUnderreplicatedFragments(lh);
                    Iterator iterator = fragments.iterator();
                    do {
                        if (!iterator.hasNext()) return;
                    } while ((fragment = (LedgerFragment)iterator.next()).isClosed());
                    lh = ReplicationWorker.this.admin.openLedger(ledgerId);
                    return;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    LOG.info("InterruptedException while replicating fragments", (Throwable)e);
                    return;
                }
                catch (BKException.BKNoSuchLedgerExistsException bknsle) {
                    if (!LOG.isDebugEnabled()) return;
                    LOG.debug("Ledger was deleted, safe to continue", (Throwable)bknsle);
                    return;
                }
                catch (BKException e) {
                    LOG.error("BKException while fencing the ledger for rereplication of postponed ledgers", (Throwable)e);
                    return;
                }
                finally {
                    try {
                        if (lh != null) {
                            lh.close();
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        LOG.info("InterruptedException while closing ledger", (Throwable)e);
                    }
                    catch (BKException e) {
                        LOG.warn("BKException while closing ledger ", (Throwable)e);
                    }
                    finally {
                        try {
                            ReplicationWorker.this.underreplicationManager.releaseUnderreplicatedLedger(ledgerId);
                        }
                        catch (ReplicationException.UnavailableException e) {
                            LOG.error("UnavailableException while replicating fragments", (Throwable)e);
                            ReplicationWorker.this.shutdown();
                        }
                    }
                }
            }
        };
        this.pendingReplicationTimer.schedule(timerTask, gracePeriod);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        LOG.info("Shutting down replication worker");
        ReplicationWorker replicationWorker = this;
        synchronized (replicationWorker) {
            if (!this.workerRunning) {
                return;
            }
            this.workerRunning = false;
        }
        LOG.info("Shutting down ReplicationWorker");
        this.pendingReplicationTimer.cancel();
        try {
            this.workerThread.interrupt();
            this.workerThread.join();
        }
        catch (InterruptedException e) {
            LOG.error("Interrupted during shutting down replication worker : ", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        try {
            this.bkc.close();
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted while closing the Bookie client", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        catch (BKException e) {
            LOG.warn("Exception while closing the Bookie client", (Throwable)e);
        }
        try {
            this.underreplicationManager.close();
        }
        catch (ReplicationException.UnavailableException e) {
            LOG.warn("Exception while closing the ZkLedgerUnderrepliationManager", (Throwable)e);
        }
    }

    boolean isRunning() {
        return this.workerRunning && this.workerThread.isAlive();
    }

    boolean isInReadOnlyMode() {
        return this.isInReadOnlyMode;
    }

    private boolean isTargetBookieExistsInFragmentEnsemble(LedgerHandle lh, LedgerFragment ledgerFragment) {
        List<BookieSocketAddress> ensemble = ledgerFragment.getEnsemble();
        for (BookieSocketAddress bkAddr : ensemble) {
            if (!this.targetBookie.equals(bkAddr)) continue;
            return true;
        }
        return false;
    }

    private static class CheckerCallback
    implements BookkeeperInternalCallbacks.GenericCallback<Set<LedgerFragment>> {
        private Set<LedgerFragment> result = null;
        private CountDownLatch latch = new CountDownLatch(1);

        private CheckerCallback() {
        }

        @Override
        public void operationComplete(int rc, Set<LedgerFragment> result) {
            this.result = result;
            this.latch.countDown();
        }

        Set<LedgerFragment> waitAndGetResult() throws InterruptedException {
            this.latch.await();
            return this.result;
        }
    }
}

