/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.ISessionsRepository;
import io.moquette.broker.MQTTConnection;
import io.moquette.broker.SessionMessageQueue;
import io.moquette.broker.SessionRegistry;
import io.moquette.broker.Utils;
import io.moquette.broker.subscriptions.Subscription;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttProperties;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.codec.mqtt.MqttReasonCodes;
import io.netty.handler.codec.mqtt.MqttSubscriptionOption;
import io.netty.handler.codec.mqtt.MqttVersion;
import java.net.InetSocketAddress;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Session {
    private static final Logger LOG = LoggerFactory.getLogger(Session.class);
    static final int INFINITE_EXPIRY = Integer.MAX_VALUE;
    private final boolean resendInflightOnTimeout;
    private Collection<Integer> nonAckPacketIds;
    private boolean clean;
    private final SessionMessageQueue<SessionRegistry.EnqueuedMessage> sessionQueue;
    private final AtomicReference<SessionStatus> status = new AtomicReference<SessionStatus>(SessionStatus.DISCONNECTED);
    private MQTTConnection mqttConnection;
    private final Set<Subscription> subscriptions = new HashSet<Subscription>();
    private final Map<Integer, SessionRegistry.EnqueuedMessage> inflightWindow = new HashMap<Integer, SessionRegistry.EnqueuedMessage>();
    private final DelayQueue<InFlightPacket> inflightTimeouts = new DelayQueue();
    private final Map<Integer, MqttPublishMessage> qos2Receiving = new HashMap<Integer, MqttPublishMessage>();
    private ISessionsRepository.SessionData data;
    private boolean resendingNonAcked = false;

    Session(ISessionsRepository.SessionData data, boolean clean, SessionMessageQueue<SessionRegistry.EnqueuedMessage> sessionQueue) {
        if (sessionQueue == null) {
            throw new IllegalArgumentException("sessionQueue parameter can't be null");
        }
        this.data = data;
        this.clean = clean;
        this.sessionQueue = sessionQueue;
        this.resendInflightOnTimeout = data.protocolVersion() != MqttVersion.MQTT_5;
    }

    public boolean expireImmediately() {
        return this.data.expiryInterval() == 0;
    }

    public void updateSessionData(ISessionsRepository.SessionData newSessionData) {
        this.data = newSessionData;
    }

    void markAsNotClean() {
        this.clean = false;
    }

    void markConnecting() {
        this.assignState(SessionStatus.DISCONNECTED, SessionStatus.CONNECTING);
    }

    boolean completeConnection() {
        return this.assignState(SessionStatus.CONNECTING, SessionStatus.CONNECTED);
    }

    void bind(MQTTConnection mqttConnection) {
        this.mqttConnection = mqttConnection;
    }

    boolean isBoundTo(MQTTConnection mqttConnection) {
        return this.mqttConnection == mqttConnection;
    }

    public boolean disconnected() {
        return this.status.get() == SessionStatus.DISCONNECTED;
    }

    public boolean connected() {
        return this.status.get() == SessionStatus.CONNECTED;
    }

    public String getClientID() {
        return this.data.clientId();
    }

    public List<Subscription> getSubscriptions() {
        return new ArrayList<Subscription>(this.subscriptions);
    }

    public void addSubscriptions(List<Subscription> newSubscriptions) {
        this.subscriptions.addAll(newSubscriptions);
    }

    public void removeSubscription(Topic topic) {
        this.subscriptions.remove(new Subscription(this.data.clientId(), topic, MqttSubscriptionOption.onlyFromQos((MqttQoS)MqttQoS.EXACTLY_ONCE)));
    }

    public boolean hasWill() {
        return this.getSessionData().hasWill();
    }

    public ISessionsRepository.Will getWill() {
        return this.getSessionData().will();
    }

    boolean assignState(SessionStatus expected, SessionStatus newState) {
        return this.status.compareAndSet(expected, newState);
    }

    public void closeImmediately() {
        this.mqttConnection.dropConnection();
        this.mqttConnection = null;
        this.status.set(SessionStatus.DISCONNECTED);
    }

    public void disconnect() {
        boolean res = this.assignState(SessionStatus.CONNECTED, SessionStatus.DISCONNECTING);
        if (!res) {
            return;
        }
        this.mqttConnection = null;
        this.updateSessionData(this.data.withoutWill());
        this.assignState(SessionStatus.DISCONNECTING, SessionStatus.DISCONNECTED);
    }

    boolean isClean() {
        return this.clean;
    }

    public void processPubRec(int pubRecPacketId) {
        this.cleanFromInflight(pubRecPacketId);
        SessionRegistry.EnqueuedMessage removed = this.inflightWindow.remove(pubRecPacketId);
        if (removed == null) {
            LOG.warn("Received a PUBREC with not matching packetId");
            return;
        }
        Utils.release(removed, "target session - phase 1 Qos2 pull from inflight");
        if (removed instanceof SessionRegistry.PubRelMarker) {
            LOG.info("Received a PUBREC for packetId that was already moved in second step of Qos2");
            return;
        }
        if (this.mqttConnection == null) {
            return;
        }
        this.inflightWindow.put(pubRecPacketId, new SessionRegistry.PubRelMarker());
        if (this.resendInflightOnTimeout) {
            this.inflightTimeouts.add(new InFlightPacket(pubRecPacketId, 5000L));
        }
        MqttMessage pubRel = MQTTConnection.pubrel(pubRecPacketId);
        this.mqttConnection.sendIfWritableElseDrop(pubRel);
        if (this.resendingNonAcked) {
            this.resendInflightNotAcked();
        } else {
            this.drainQueueToConnection();
        }
    }

    public void processPubComp(int messageID) {
        this.cleanFromInflight(messageID);
        SessionRegistry.EnqueuedMessage removed = this.inflightWindow.remove(messageID);
        if (removed == null) {
            LOG.warn("Received a PUBCOMP with not matching packetId in the inflight cache");
            return;
        }
        Utils.release(removed, "target session - phase 2 Qos2 pull from inflight");
        this.mqttConnection.sendQuota().releaseSlot();
        this.drainQueueToConnection();
    }

    public void sendRetainedPublishOnSessionAtQos(Topic topic, MqttQoS qos, ByteBuf payload, MqttProperties.MqttProperty ... mqttProperties) {
        SessionRegistry.PublishedMessage publishedMessage = new SessionRegistry.PublishedMessage(topic, qos, payload, true, Instant.MAX, mqttProperties);
        this.sendPublishOnSessionAtQos(publishedMessage);
    }

    void sendPublishOnSessionAtQos(SessionRegistry.PublishedMessage publishRequest) {
        switch (publishRequest.getPublishingQos()) {
            case AT_MOST_ONCE: {
                if (!this.connected()) break;
                this.sendPublishQos0(publishRequest);
                break;
            }
            case AT_LEAST_ONCE: {
                this.sendPublishQos1(publishRequest);
                break;
            }
            case EXACTLY_ONCE: {
                this.sendPublishQos2(publishRequest);
                break;
            }
            case FAILURE: {
                LOG.error("Not admissible");
            }
        }
    }

    private void sendPublishQos0(SessionRegistry.PublishedMessage publishRequest) {
        if (publishRequest.isExpired()) {
            LOG.debug("Sending publish at QoS0 already expired, drop it");
            return;
        }
        MqttProperties.MqttProperty[] mqttProperties = publishRequest.updatePublicationExpiryIfPresentOrAdd();
        MqttPublishMessage publishMsg = MQTTConnection.createPublishMessage(publishRequest.getTopic().toString(), publishRequest.getPublishingQos(), publishRequest.getPayload(), 0, publishRequest.retained, false, mqttProperties);
        this.mqttConnection.sendPublish(publishMsg);
    }

    private void sendPublishQos1(SessionRegistry.PublishedMessage publishRequest) {
        if (!this.connected() && this.isClean()) {
            return;
        }
        if (publishRequest.isExpired()) {
            LOG.debug("Sending publish at QoS1 already expired, expected to happen before {}, drop it", (Object)publishRequest.messageExpiry);
            return;
        }
        MQTTConnection localMqttConnectionRef = this.mqttConnection;
        this.sendPublishInFlightWindowOrQueueing(localMqttConnectionRef, publishRequest);
    }

    private void sendPublishQos2(SessionRegistry.PublishedMessage publishRequest) {
        if (publishRequest.isExpired()) {
            LOG.debug("Sending publish at QoS2 already expired, drop it");
            return;
        }
        MQTTConnection localMqttConnectionRef = this.mqttConnection;
        this.sendPublishInFlightWindowOrQueueing(localMqttConnectionRef, publishRequest);
    }

    private void sendPublishInFlightWindowOrQueueing(MQTTConnection localMqttConnectionRef, SessionRegistry.PublishedMessage publishRequest) {
        Utils.retain(publishRequest, "target session - forward to inflight or queue");
        if (this.canSkipQueue(localMqttConnectionRef)) {
            this.mqttConnection.sendQuota().consumeSlot();
            int packetId = localMqttConnectionRef.nextPacketId();
            LOG.debug("Adding into inflight for session {} at QoS {}", (Object)this.getClientID(), (Object)publishRequest.getPublishingQos());
            SessionRegistry.EnqueuedMessage old = this.inflightWindow.put(packetId, publishRequest);
            if (old != null) {
                Utils.release(old, "target session - replace existing slot");
                this.mqttConnection.sendQuota().releaseSlot();
            }
            if (this.resendInflightOnTimeout) {
                this.inflightTimeouts.add(new InFlightPacket(packetId, 5000L));
            }
            MqttProperties.MqttProperty[] mqttProperties = publishRequest.updatePublicationExpiryIfPresentOrAdd();
            MqttPublishMessage publishMsg = MQTTConnection.createPublishMessage(publishRequest.topic.toString(), publishRequest.getPublishingQos(), publishRequest.payload, packetId, publishRequest.retained, false, mqttProperties);
            localMqttConnectionRef.sendPublish(publishMsg);
            this.drainQueueToConnection();
        } else {
            this.sessionQueue.enqueue(publishRequest);
            LOG.debug("Enqueue to peer session {} at QoS {}", (Object)this.getClientID(), (Object)publishRequest.getPublishingQos());
        }
    }

    private boolean canSkipQueue(MQTTConnection localMqttConnectionRef) {
        return localMqttConnectionRef != null && this.sessionQueue.isEmpty() && this.mqttConnection.sendQuota().hasFreeSlots() && this.connected() && localMqttConnectionRef.channel.isWritable();
    }

    private boolean inflightHasSlotsAndConnectionIsUp() {
        return this.mqttConnection.sendQuota().hasFreeSlots() && this.connected() && this.mqttConnection.channel.isWritable();
    }

    void pubAckReceived(int ackPacketId) {
        this.cleanFromInflight(ackPacketId);
        SessionRegistry.EnqueuedMessage removed = this.inflightWindow.remove(ackPacketId);
        if (removed == null) {
            LOG.warn("Received a PUBACK with not matching packetId({}) in the inflight cache({})", (Object)ackPacketId, this.inflightWindow.keySet());
            return;
        }
        Utils.release(removed, "target session - inflight remove");
        this.mqttConnection.sendQuota().releaseSlot();
        LOG.debug("Received PUBACK {} for session {}", (Object)ackPacketId, (Object)this.getClientID());
        if (this.resendingNonAcked) {
            this.resendInflightNotAcked();
        } else {
            this.drainQueueToConnection();
        }
    }

    private void cleanFromInflight(int ackPacketId) {
        this.inflightTimeouts.removeIf(d -> d.packetId == ackPacketId);
    }

    public void flushAllQueuedMessages() {
        this.drainQueueToConnection();
    }

    public void resendInflightNotAcked() {
        if (!this.resendingNonAcked) {
            if (this.resendInflightOnTimeout) {
                ArrayList expired = new ArrayList();
                this.inflightTimeouts.drainTo(expired);
                this.nonAckPacketIds = expired.stream().map(p -> p.packetId).collect(Collectors.toList());
            } else {
                this.nonAckPacketIds = new ArrayList<Integer>(this.inflightWindow.keySet());
            }
            this.debugLogPacketIds(this.nonAckPacketIds);
        }
        if (this.nonAckPacketIds.size() > this.mqttConnection.sendQuota().availableSlots()) {
            this.resendingNonAcked = true;
            List<Integer> partition = this.nonAckPacketIds.stream().limit(this.mqttConnection.sendQuota().availableSlots()).collect(Collectors.toList());
            this.resendNonAckedIdsPartition(partition);
            for (Integer id : partition) {
                this.nonAckPacketIds.remove(id);
            }
        } else {
            this.resendNonAckedIdsPartition(this.nonAckPacketIds);
            this.resendingNonAcked = false;
        }
    }

    private void resendNonAckedIdsPartition(Collection<Integer> packetIdsToResend) {
        for (Integer notAckPacketId : packetIdsToResend) {
            SessionRegistry.EnqueuedMessage msg = this.inflightWindow.get(notAckPacketId);
            if (msg == null) continue;
            if (msg instanceof SessionRegistry.PubRelMarker) {
                MqttMessage pubRel = MQTTConnection.pubrel(notAckPacketId);
                if (this.resendInflightOnTimeout) {
                    this.inflightTimeouts.add(new InFlightPacket(notAckPacketId, 5000L));
                }
                this.mqttConnection.sendIfWritableElseDrop(pubRel);
                continue;
            }
            SessionRegistry.PublishedMessage pubMsg = (SessionRegistry.PublishedMessage)msg;
            Topic topic = pubMsg.topic;
            MqttQoS qos = pubMsg.publishingQos;
            ByteBuf payload = pubMsg.payload;
            MqttProperties.MqttProperty[] mqttProperties = pubMsg.mqttProperties;
            MqttPublishMessage publishMsg = MQTTConnection.createNotRetainedDuplicatedPublishMessage(notAckPacketId, topic, qos, payload, mqttProperties);
            if (this.resendInflightOnTimeout) {
                this.inflightTimeouts.add(new InFlightPacket(notAckPacketId, 5000L));
            }
            this.mqttConnection.sendPublish(publishMsg);
            this.mqttConnection.sendQuota().consumeSlot();
        }
    }

    private void debugLogPacketIds(Collection<Integer> packetIds) {
        if (!LOG.isDebugEnabled() || packetIds.isEmpty()) {
            return;
        }
        StringBuilder sb = new StringBuilder();
        for (Integer packetId : packetIds) {
            sb.append(packetId).append(", ");
        }
        LOG.debug("Resending {} in flight packets [{}]", (Object)packetIds.size(), (Object)sb);
    }

    private void drainQueueToConnection() {
        while (this.connected() && !this.sessionQueue.isEmpty() && this.inflightHasSlotsAndConnectionIsUp()) {
            SessionRegistry.EnqueuedMessage msg = this.sessionQueue.dequeue();
            if (msg == null) {
                return;
            }
            SessionRegistry.PublishedMessage msgPub = (SessionRegistry.PublishedMessage)msg;
            if (msgPub.isExpired()) {
                LOG.debug("Drop an expired message contained in the queue");
                return;
            }
            this.mqttConnection.sendQuota().consumeSlot();
            int sendPacketId = this.mqttConnection.nextPacketId();
            SessionRegistry.EnqueuedMessage old = this.inflightWindow.put(sendPacketId, msg);
            if (old != null) {
                Utils.release(old, "target session - drain queue push to inflight");
                this.mqttConnection.sendQuota().releaseSlot();
            }
            if (this.resendInflightOnTimeout) {
                this.inflightTimeouts.add(new InFlightPacket(sendPacketId, 5000L));
            }
            MqttProperties.MqttProperty[] mqttProperties = msgPub.updatePublicationExpiryIfPresentOrAdd();
            MqttPublishMessage publishMsg = MQTTConnection.createNotRetainedPublishMessage(msgPub.topic.toString(), msgPub.publishingQos, msgPub.payload, sendPacketId, mqttProperties);
            this.mqttConnection.sendPublish(publishMsg);
        }
    }

    public void writabilityChanged() {
        this.drainQueueToConnection();
    }

    public void reconnectSession() {
        LOG.trace("Republishing all saved messages for session {}", (Object)this);
        this.resendInflightNotAcked();
        if (!this.resendingNonAcked) {
            this.drainQueueToConnection();
        }
    }

    public void receivedPublishQos2(int messageID, MqttPublishMessage msg) {
        Utils.retain(msg, "phase 2 qos2");
        MqttPublishMessage old = this.qos2Receiving.put(messageID, msg);
        Utils.release(old, "phase 2 qos2 - packet id duplicated");
    }

    public void receivedPubRelQos2(int messageID) {
        MqttPublishMessage removedMsg = this.qos2Receiving.remove(messageID);
        Utils.release(removedMsg, "phase 2 qos2");
    }

    Optional<InetSocketAddress> remoteAddress() {
        if (this.connected()) {
            return Optional.of(this.mqttConnection.remoteAddress());
        }
        return Optional.empty();
    }

    public void cleanUp() {
        this.sessionQueue.closeAndPurge();
        this.inflightTimeouts.clear();
        for (SessionRegistry.EnqueuedMessage enqueuedMessage : this.inflightWindow.values()) {
            Utils.release(enqueuedMessage, "session cleanup - inflight window");
        }
        for (MqttPublishMessage mqttPublishMessage : this.qos2Receiving.values()) {
            Utils.release(mqttPublishMessage, "session cleanup - phase 2 cache");
        }
    }

    ISessionsRepository.SessionData getSessionData() {
        return this.data;
    }

    public void disconnectFromBroker() {
        this.mqttConnection.brokerDisconnect(MqttReasonCodes.Disconnect.MALFORMED_PACKET);
        this.disconnect();
    }

    public String toString() {
        return "Session{clientId='" + this.data.clientId() + '\'' + ", clean=" + this.clean + ", status=" + this.status + ", inflightSlots=" + this.mqttConnection.sendQuota().availableSlots() + '}';
    }

    static enum SessionStatus {
        CONNECTED,
        CONNECTING,
        DISCONNECTING,
        DISCONNECTED,
        DESTROYED;

    }

    static class InFlightPacket
    implements Delayed {
        final int packetId;
        private long startTime;

        InFlightPacket(int packetId, long delayInMilliseconds) {
            this.packetId = packetId;
            this.startTime = System.currentTimeMillis() + delayInMilliseconds;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long diff = this.startTime - System.currentTimeMillis();
            return unit.convert(diff, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            if (this.startTime - ((InFlightPacket)o).startTime == 0L) {
                return 0;
            }
            if (this.startTime - ((InFlightPacket)o).startTime > 0L) {
                return 1;
            }
            return -1;
        }
    }
}

