/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.sieve.jpa;

import com.github.fge.lambdas.Throwing;
import com.google.common.collect.ImmutableList;
import jakarta.inject.Inject;
import jakarta.persistence.EntityManager;
import jakarta.persistence.EntityManagerFactory;
import jakarta.persistence.EntityTransaction;
import jakarta.persistence.NoResultException;
import jakarta.persistence.PersistenceException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;
import org.apache.commons.io.IOUtils;
import org.apache.james.backends.jpa.TransactionRunner;
import org.apache.james.core.Username;
import org.apache.james.core.quota.QuotaSizeLimit;
import org.apache.james.core.quota.QuotaSizeUsage;
import org.apache.james.sieve.jpa.model.JPASieveQuota;
import org.apache.james.sieve.jpa.model.JPASieveScript;
import org.apache.james.sieverepository.api.ScriptContent;
import org.apache.james.sieverepository.api.ScriptName;
import org.apache.james.sieverepository.api.ScriptSummary;
import org.apache.james.sieverepository.api.SieveRepository;
import org.apache.james.sieverepository.api.exception.DuplicateException;
import org.apache.james.sieverepository.api.exception.IsActiveException;
import org.apache.james.sieverepository.api.exception.QuotaExceededException;
import org.apache.james.sieverepository.api.exception.QuotaNotFoundException;
import org.apache.james.sieverepository.api.exception.ScriptNotFoundException;
import org.apache.james.sieverepository.api.exception.StorageException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class JPASieveRepository
implements SieveRepository {
    private static final Logger LOGGER = LoggerFactory.getLogger(JPASieveRepository.class);
    private static final String DEFAULT_SIEVE_QUOTA_USERNAME = "default.quota";
    private final TransactionRunner transactionRunner;
    private final EntityManagerFactory entityManagerFactory;

    @Inject
    public JPASieveRepository(EntityManagerFactory entityManagerFactory) {
        this.transactionRunner = new TransactionRunner(entityManagerFactory);
        this.entityManagerFactory = entityManagerFactory;
    }

    public void haveSpace(Username username, ScriptName name, long size) throws QuotaExceededException, StorageException {
        QuotaSizeLimit quota;
        long usedSpace = this.findAllSieveScriptsForUser(username).stream().filter(sieveScript -> !sieveScript.getScriptName().equals(name.getValue())).mapToLong(JPASieveScript::getScriptSize).sum();
        if (this.overQuotaAfterModification(usedSpace, size, quota = this.limitToUser(username))) {
            throw new QuotaExceededException();
        }
    }

    private QuotaSizeLimit limitToUser(Username username) throws StorageException {
        return this.findQuotaForUser(username.asString()).or(Throwing.supplier(() -> this.findQuotaForUser(DEFAULT_SIEVE_QUOTA_USERNAME)).sneakyThrow()).map(JPASieveQuota::toQuotaSize).orElse(QuotaSizeLimit.unlimited());
    }

    private boolean overQuotaAfterModification(long usedSpace, long size, QuotaSizeLimit quota) {
        return QuotaSizeUsage.size((long)usedSpace).add(size).exceedLimit(quota);
    }

    public void putScript(Username username, ScriptName name, ScriptContent content) throws StorageException, QuotaExceededException {
        this.transactionRunner.runAndHandleException(Throwing.consumer(entityManager -> {
            try {
                this.haveSpace(username, name, content.length());
                JPASieveScript jpaSieveScript = JPASieveScript.builder().username(username.asString()).scriptName(name.getValue()).scriptContent(content).build();
                entityManager.persist((Object)jpaSieveScript);
            }
            catch (QuotaExceededException | StorageException e) {
                this.rollbackTransactionIfActive(entityManager.getTransaction());
                throw e;
            }
        }).sneakyThrow(), this.throwStorageExceptionConsumer("Unable to put script for user " + username.asString()));
    }

    public List<ScriptSummary> listScripts(Username username) throws StorageException {
        return (List)this.findAllSieveScriptsForUser(username).stream().map(JPASieveScript::toSummary).collect(ImmutableList.toImmutableList());
    }

    public Flux<ScriptSummary> listScriptsReactive(Username username) {
        return Mono.fromCallable(() -> this.listScripts(username)).flatMapMany(Flux::fromIterable);
    }

    public Stream<JPASieveScript> listAllSieveScripts() {
        EntityManager entityManager = this.entityManagerFactory.createEntityManager();
        return entityManager.createNamedQuery("listAllSieveScripts").getResultStream();
    }

    public Stream<JPASieveQuota> listAllSieveQuotas() {
        EntityManager entityManager = this.entityManagerFactory.createEntityManager();
        return entityManager.createNamedQuery("listAllSieveQuotas").getResultStream();
    }

    private List<JPASieveScript> findAllSieveScriptsForUser(Username username) throws StorageException {
        return (List)this.transactionRunner.runAndRetrieveResult(entityManager -> {
            List sieveScripts = entityManager.createNamedQuery("findAllByUsername", JPASieveScript.class).setParameter("username", (Object)username.asString()).getResultList();
            return Optional.ofNullable(sieveScripts).orElse((List)ImmutableList.of());
        }, this.throwStorageException("Unable to list scripts for user " + username.asString()));
    }

    public ZonedDateTime getActivationDateForActiveScript(Username username) throws StorageException, ScriptNotFoundException {
        Optional<JPASieveScript> script = this.findActiveSieveScript(username);
        JPASieveScript activeSieveScript = script.orElseThrow(() -> new ScriptNotFoundException("Unable to find active script for user " + username.asString()));
        return activeSieveScript.getActivationDateTime().toZonedDateTime();
    }

    public InputStream getActive(Username username) throws ScriptNotFoundException, StorageException {
        Optional<JPASieveScript> script = this.findActiveSieveScript(username);
        JPASieveScript activeSieveScript = script.orElseThrow(() -> new ScriptNotFoundException("Unable to find active script for user " + username.asString()));
        return IOUtils.toInputStream((String)activeSieveScript.getScriptContent(), (Charset)StandardCharsets.UTF_8);
    }

    private Optional<JPASieveScript> findActiveSieveScript(Username username) throws StorageException {
        return (Optional)this.transactionRunner.runAndRetrieveResult(Throwing.function(entityManager -> this.findActiveSieveScript(username, (EntityManager)entityManager)).sneakyThrow(), this.throwStorageException("Unable to find active script for user " + username.asString()));
    }

    private Optional<JPASieveScript> findActiveSieveScript(Username username, EntityManager entityManager) throws StorageException {
        try {
            JPASieveScript activeSieveScript = (JPASieveScript)entityManager.createNamedQuery("findActiveByUsername", JPASieveScript.class).setParameter("username", (Object)username.asString()).getSingleResult();
            return Optional.ofNullable(activeSieveScript);
        }
        catch (NoResultException e) {
            LOGGER.debug("Sieve script not found for user {}", (Object)username.asString());
            return Optional.empty();
        }
    }

    public void setActive(Username username, ScriptName name) throws ScriptNotFoundException, StorageException {
        this.transactionRunner.runAndHandleException(Throwing.consumer(entityManager -> {
            try {
                if (SieveRepository.NO_SCRIPT_NAME.equals((Object)name)) {
                    this.switchOffActiveScript(username, (EntityManager)entityManager);
                } else {
                    this.setActiveScript(username, name, (EntityManager)entityManager);
                }
            }
            catch (ScriptNotFoundException | StorageException e) {
                this.rollbackTransactionIfActive(entityManager.getTransaction());
                throw e;
            }
        }).sneakyThrow(), this.throwStorageExceptionConsumer("Unable to set active script " + name.getValue() + " for user " + username.asString()));
    }

    private void switchOffActiveScript(Username username, EntityManager entityManager) throws StorageException {
        Optional<JPASieveScript> activeSieveScript = this.findActiveSieveScript(username, entityManager);
        activeSieveScript.ifPresent(JPASieveScript::deactivate);
    }

    private void setActiveScript(Username username, ScriptName name, EntityManager entityManager) throws StorageException, ScriptNotFoundException {
        JPASieveScript sieveScript = this.findSieveScript(username, name, entityManager).orElseThrow(() -> new ScriptNotFoundException("Unable to find script " + name.getValue() + " for user " + username.asString()));
        this.findActiveSieveScript(username, entityManager).ifPresent(JPASieveScript::deactivate);
        sieveScript.activate();
    }

    public InputStream getScript(Username username, ScriptName name) throws ScriptNotFoundException, StorageException {
        Optional<JPASieveScript> script = this.findSieveScript(username, name);
        JPASieveScript sieveScript = script.orElseThrow(() -> new ScriptNotFoundException("Unable to find script " + name.getValue() + " for user " + username.asString()));
        return IOUtils.toInputStream((String)sieveScript.getScriptContent(), (Charset)StandardCharsets.UTF_8);
    }

    private Optional<JPASieveScript> findSieveScript(Username username, ScriptName scriptName) throws StorageException {
        return (Optional)this.transactionRunner.runAndRetrieveResult(entityManager -> this.findSieveScript(username, scriptName, (EntityManager)entityManager), this.throwStorageException("Unable to find script " + scriptName.getValue() + " for user " + username.asString()));
    }

    private Optional<JPASieveScript> findSieveScript(Username username, ScriptName scriptName, EntityManager entityManager) {
        try {
            JPASieveScript sieveScript = (JPASieveScript)entityManager.createNamedQuery("findSieveScript", JPASieveScript.class).setParameter("username", (Object)username.asString()).setParameter("scriptName", (Object)scriptName.getValue()).getSingleResult();
            return Optional.ofNullable(sieveScript);
        }
        catch (NoResultException e) {
            LOGGER.debug("Sieve script not found for user {}", (Object)username.asString());
            return Optional.empty();
        }
    }

    public void deleteScript(Username username, ScriptName name) throws ScriptNotFoundException, IsActiveException, StorageException {
        this.transactionRunner.runAndHandleException(Throwing.consumer(entityManager -> {
            Optional<JPASieveScript> sieveScript = this.findSieveScript(username, name, (EntityManager)entityManager);
            if (!sieveScript.isPresent()) {
                this.rollbackTransactionIfActive(entityManager.getTransaction());
                throw new ScriptNotFoundException("Unable to find script " + name.getValue() + " for user " + username.asString());
            }
            JPASieveScript sieveScriptToRemove = sieveScript.get();
            if (sieveScriptToRemove.isActive()) {
                this.rollbackTransactionIfActive(entityManager.getTransaction());
                throw new IsActiveException("Unable to delete active script " + name.getValue() + " for user " + username.asString());
            }
            entityManager.remove((Object)sieveScriptToRemove);
        }).sneakyThrow(), this.throwStorageExceptionConsumer("Unable to delete script " + name.getValue() + " for user " + username.asString()));
    }

    public void renameScript(Username username, ScriptName oldName, ScriptName newName) throws ScriptNotFoundException, DuplicateException, StorageException {
        this.transactionRunner.runAndHandleException(Throwing.consumer(entityManager -> {
            Optional<JPASieveScript> sieveScript = this.findSieveScript(username, oldName, (EntityManager)entityManager);
            if (!sieveScript.isPresent()) {
                this.rollbackTransactionIfActive(entityManager.getTransaction());
                throw new ScriptNotFoundException("Unable to find script " + oldName.getValue() + " for user " + username.asString());
            }
            Optional<JPASieveScript> duplicatedSieveScript = this.findSieveScript(username, newName, (EntityManager)entityManager);
            if (duplicatedSieveScript.isPresent()) {
                this.rollbackTransactionIfActive(entityManager.getTransaction());
                throw new DuplicateException("Unable to rename script. Duplicate found " + newName.getValue() + " for user " + username.asString());
            }
            JPASieveScript sieveScriptToRename = sieveScript.get();
            sieveScriptToRename.renameTo(newName);
        }).sneakyThrow(), this.throwStorageExceptionConsumer("Unable to rename script " + oldName.getValue() + " for user " + username.asString()));
    }

    private void rollbackTransactionIfActive(EntityTransaction transaction) {
        if (transaction.isActive()) {
            transaction.rollback();
        }
    }

    public boolean hasDefaultQuota() throws StorageException {
        Optional<JPASieveQuota> defaultQuota = this.findQuotaForUser(DEFAULT_SIEVE_QUOTA_USERNAME);
        return defaultQuota.isPresent();
    }

    public QuotaSizeLimit getDefaultQuota() throws QuotaNotFoundException, StorageException {
        JPASieveQuota jpaSieveQuota = this.findQuotaForUser(DEFAULT_SIEVE_QUOTA_USERNAME).orElseThrow(() -> new QuotaNotFoundException("Unable to find quota for default user"));
        return QuotaSizeLimit.size((long)jpaSieveQuota.getSize());
    }

    public void setDefaultQuota(QuotaSizeLimit quota) throws StorageException {
        this.setQuotaForUser(DEFAULT_SIEVE_QUOTA_USERNAME, quota);
    }

    public void removeQuota() throws QuotaNotFoundException, StorageException {
        this.removeQuotaForUser(DEFAULT_SIEVE_QUOTA_USERNAME);
    }

    public boolean hasQuota(Username username) throws StorageException {
        Optional<JPASieveQuota> quotaForUser = this.findQuotaForUser(username.asString());
        return quotaForUser.isPresent();
    }

    public QuotaSizeLimit getQuota(Username username) throws QuotaNotFoundException, StorageException {
        JPASieveQuota jpaSieveQuota = this.findQuotaForUser(username.asString()).orElseThrow(() -> new QuotaNotFoundException("Unable to find quota for user " + username.asString()));
        return QuotaSizeLimit.size((long)jpaSieveQuota.getSize());
    }

    public void setQuota(Username username, QuotaSizeLimit quota) throws StorageException {
        this.setQuotaForUser(username.asString(), quota);
    }

    public void removeQuota(Username username) throws QuotaNotFoundException, StorageException {
        this.removeQuotaForUser(username.asString());
    }

    private Optional<JPASieveQuota> findQuotaForUser(String username) throws StorageException {
        return (Optional)this.transactionRunner.runAndRetrieveResult(entityManager -> this.findQuotaForUser(username, (EntityManager)entityManager), this.throwStorageException("Unable to find quota for user " + username));
    }

    private <T> Function<PersistenceException, T> throwStorageException(String message) {
        return Throwing.function(e -> {
            throw new StorageException(message, (Throwable)e);
        }).sneakyThrow();
    }

    private Consumer<PersistenceException> throwStorageExceptionConsumer(String message) {
        return Throwing.consumer(e -> {
            throw new StorageException(message, (Throwable)e);
        }).sneakyThrow();
    }

    private Optional<JPASieveQuota> findQuotaForUser(String username, EntityManager entityManager) {
        try {
            JPASieveQuota sieveQuota = (JPASieveQuota)entityManager.createNamedQuery("findByUsername", JPASieveQuota.class).setParameter("username", (Object)username).getSingleResult();
            return Optional.of(sieveQuota);
        }
        catch (NoResultException e) {
            return Optional.empty();
        }
    }

    private void setQuotaForUser(String username, QuotaSizeLimit quota) throws StorageException {
        this.transactionRunner.runAndHandleException((Consumer)Throwing.consumer(entityManager -> {
            Optional<JPASieveQuota> sieveQuota = this.findQuotaForUser(username, (EntityManager)entityManager);
            if (sieveQuota.isPresent()) {
                JPASieveQuota jpaSieveQuota = sieveQuota.get();
                jpaSieveQuota.setSize(quota);
                entityManager.merge((Object)jpaSieveQuota);
            } else {
                JPASieveQuota jpaSieveQuota = new JPASieveQuota(username, quota.asLong());
                entityManager.persist((Object)jpaSieveQuota);
            }
        }), this.throwStorageExceptionConsumer("Unable to set quota for user " + username));
    }

    private void removeQuotaForUser(String username) throws StorageException {
        this.transactionRunner.runAndHandleException((Consumer)Throwing.consumer(entityManager -> {
            Optional<JPASieveQuota> quotaForUser = this.findQuotaForUser(username, (EntityManager)entityManager);
            quotaForUser.ifPresent(arg_0 -> ((EntityManager)entityManager).remove(arg_0));
        }), this.throwStorageExceptionConsumer("Unable to remove quota for user " + username));
    }

    public Mono<Void> resetSpaceUsedReactive(Username username, long spaceUsed) {
        return Mono.error((Throwable)new UnsupportedOperationException());
    }
}

