/****************************************************************************
 * arch/risc-v/src/common/riscv_cpustart.c
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <stdint.h>
#include <assert.h>
#include <debug.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include <nuttx/arch.h>
#include <nuttx/spinlock.h>
#include <nuttx/sched_note.h>

#include "sched/sched.h"
#include "init/init.h"
#include "riscv_internal.h"

#ifdef CONFIG_BUILD_KERNEL
#  include "riscv_mmu.h"
#endif

#include "chip.h"

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: riscv_cpu_boot
 *
 * Description:
 *   Boot handler for cpu[x]
 *
 * Input Parameters:
 *   None
 *
 * Returned Value:
 *   None
 *
 ****************************************************************************/

void riscv_cpu_boot(int cpu)
{
  /* Clear IPI for CPU(cpu) */

  putreg32(0, (uintptr_t)RISCV_IPI + (4 * cpu));

  /* Enable machine software interrupt for IPI to boot */

  up_enable_irq(RISCV_IRQ_SOFT);

  /* Wait interrupt */

  asm("WFI");

#ifdef CONFIG_BUILD_KERNEL
  /* Initialize the per CPU areas */

  riscv_percpu_add_hart((uintptr_t)cpu);

  /* Enable MMU */

  binfo("mmu_enable: satp=%lx\n", g_kernel_pgt_pbase);
  mmu_enable(g_kernel_pgt_pbase, 0);
#endif

  _info("CPU%d Started\n", this_cpu());

#ifdef CONFIG_STACK_COLORATION
  struct tcb_s *tcb = this_task();

  /* If stack debug is enabled, then fill the stack with a
   * recognizable value that we can use later to test for high
   * water marks.
   */

  riscv_stack_color(tcb->stack_alloc_ptr, 0);
#endif

  /* TODO: Setup FPU */

  /* Clear machine software interrupt for CPU(cpu) */

  putreg32(0, (uintptr_t)RISCV_IPI + (4 * cpu));

#ifdef CONFIG_SCHED_INSTRUMENTATION
  /* Notify that this CPU has started */

  sched_note_cpu_started(this_task());
#endif

  up_irq_enable();

  /* Then transfer control to the IDLE task */

  nx_idle_trampoline();
}

/****************************************************************************
 * Name: up_cpu_start
 *
 * Description:
 *   In an SMP configuration, only one CPU is initially active (CPU 0).
 *   System initialization occurs on that single thread. At the completion of
 *   the initialization of the OS, just before beginning normal multitasking,
 *   the additional CPUs would be started by calling this function.
 *
 *   Each CPU is provided the entry point to its IDLE task when started.  A
 *   TCB for each CPU's IDLE task has been initialized and placed in the
 *   CPU's g_assignedtasks[cpu] list.  No stack has been allocated or
 *   initialized.
 *
 *   The OS initialization logic calls this function repeatedly until each
 *   CPU has been started, 1 through (CONFIG_SMP_NCPUS-1).
 *
 * Input Parameters:
 *   cpu - The index of the CPU being started.  This will be a numeric
 *         value in the range of one to (CONFIG_SMP_NCPUS-1).
 *         (CPU 0 is already active)
 *
 * Returned Value:
 *   Zero on success; a negated errno value on failure.
 *
 ****************************************************************************/

int up_cpu_start(int cpu)
{
  _info("CPU=%d\n", cpu);

#ifdef CONFIG_SCHED_INSTRUMENTATION
  /* Notify of the start event */

  sched_note_cpu_start(this_task(), cpu);
#endif

  /* Send IPI to CPU(cpu) */

  putreg32(1, (uintptr_t)RISCV_IPI + (cpu * 4));

  return 0;
}
