use crate::app::App;
use crate::article_view::Webview;
use crate::config::APP_ID;
use crate::content_page::ArticleViewColumn;
use crate::gobject_models::GEnclosure;
use crate::i18n::i18n;
use crate::image_widget::ImageWidget;
use crate::infrastructure::TokioRuntime;
use crate::main_window::MainWindow;
use crate::util::{GtkUtil, constants};
use clapper::{MediaItem, Mpris, PlayerState};
use clapper_gtk::{Billboard, Video};
use glib::{ControlFlow, Object, Properties, SourceId, clone, prelude::*, subclass::*};
use gstreamer::tags::{Artist, Title};
use gstreamer::{TagList, TagMergeMode};
use gtk4::{
    Accessible, Buildable, CompositeTemplate, ConstraintTarget, Overlay, Widget, prelude::*, subclass::prelude::*,
};
use libadwaita::{Bin, subclass::prelude::*};
use news_flash::models::Enclosure;
use std::cell::{Cell, RefCell};
use std::rc::Rc;
use std::time::Duration;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::VideoWidget)]
    #[template(file = "data/resources/ui_templates/media/video_widget.blp")]
    pub struct VideoWidget {
        #[template_child]
        pub video: TemplateChild<Video>,
        #[template_child]
        pub billboard: TemplateChild<Billboard>,
        #[template_child]
        pub image: TemplateChild<ImageWidget>,
        #[template_child]
        pub image_overlay: TemplateChild<Overlay>,

        #[property(get, set, name = "can-seek")]
        pub can_seek: Cell<bool>,

        #[property(get, set, construct_only)]
        pub author: RefCell<String>,

        #[property(get, set, construct_only)]
        pub title: RefCell<String>,

        #[property(get, set, construct_only)]
        pub enclosure: RefCell<GEnclosure>,

        #[property(get, set, construct_only)]
        pub item: RefCell<MediaItem>,

        pub skip_cooldown_source: Rc<RefCell<Option<SourceId>>>,
        pub skip_seconds: Rc<Cell<f64>>,

        pub rewind_cooldown_source: Rc<RefCell<Option<SourceId>>>,
        pub rewind_seconds: Rc<Cell<f64>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for VideoWidget {
        const NAME: &'static str = "VideoWidget";
        type ParentType = Bin;
        type Type = super::VideoWidget;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();

            klass.install_action("skip", None, |video_widget, _b, _c| {
                tracing::debug!("skip");
                if video_widget.can_seek() {
                    video_widget.imp().start_skip_cooldown();
                }
            });

            klass.install_action("rewind", None, |video_widget, _b, _c| {
                tracing::debug!("rewind");
                if video_widget.can_seek() {
                    video_widget.imp().start_rewind_cooldown();
                }
            });

            klass.install_action("leave-fullscreen", None, |video_widget, _b, _c| {
                tracing::debug!("leave-fullscreen");
                video_widget.add_css_class("video-widget");
                Webview::instance().set_is_enclosure_fullscreen(false);
                MainWindow::instance().set_fullscreened(false);
            });
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for VideoWidget {
        fn constructed(&self) {
            let Some(player) = self.video.player() else {
                return;
            };

            App::default()
                .bind_property("volume", &player, "volume")
                .bidirectional()
                .sync_create()
                .build();

            player.connect_state_notify(clone!(
                #[weak(rename_to = imp)]
                self,
                move |player| {
                    if player.state() == PlayerState::Playing {
                        let position = imp.enclosure.borrow().position();
                        if position > 0.0 {
                            imp.enclosure.borrow().set_position(0.0);
                            player.seek(position);
                        }
                    }

                    if player.state() == PlayerState::Paused {
                        imp.billboard.pin_message("media-playback-pause-symbolic", "Paused");
                    } else {
                        imp.billboard.unpin_pinned_message();
                    }
                }
            ));

            player.connect_seek_done(clone!(
                #[weak(rename_to = obj)]
                self.obj(),
                move |_player| {
                    obj.set_can_seek(true);
                }
            ));

            let title = self.title.borrow().clone();
            let author = self.author.borrow().clone();
            let mut tags = TagList::new();
            tags.make_mut().add::<Title>(&title.as_str(), TagMergeMode::Replace);
            tags.make_mut().add::<Artist>(&author.as_str(), TagMergeMode::Replace);
            let tag_string = tags.to_string();
            let tag_string = tag_string
                .strip_prefix("taglist, ")
                .expect("serialized GstTagList should start with `taglist, `")
                .to_string();
            tracing::debug!(%tag_string, "tags");

            let bin_description = format!("scaletempo ! taginject scope=global tags=\"{tag_string}\"");
            if let Ok(audio_filter_bin) = gstreamer::parse::bin_from_description(&bin_description, true) {
                player.set_audio_filter(Some(&audio_filter_bin));
            }

            let mpris = Mpris::new("org.mpris.MediaPlayer2.Newsflash", "Newsflash", Some(APP_ID));
            mpris.set_fallback_art_url(self.enclosure.borrow().thumbnail_url().as_deref());

            player.add_feature(&mpris);

            if let Some(queue) = player.queue() {
                queue.add_item(&self.item.borrow());
                queue.select_item(Some(&*self.item.borrow()));
            }

            let thumbnail_url = self.enclosure.borrow().thumbnail_url();
            let article_id = self.enclosure.borrow().article_id();
            if let Some(thumbnail_url) = thumbnail_url {
                self.image_overlay.set_visible(true);
                self.image.load(article_id.as_ref(), &thumbnail_url);
            }
        }

        fn dispose(&self) {
            self.stop();
        }
    }

    impl WidgetImpl for VideoWidget {}

    impl BinImpl for VideoWidget {}

    #[gtk4::template_callbacks]
    impl VideoWidget {
        #[template_callback]
        fn toggle_fullscreen(&self) {
            let obj = self.obj();
            let window = MainWindow::instance();
            let is_fullscreen = !window.is_fullscreen();

            if is_fullscreen {
                obj.remove_css_class("video-widget");
            } else {
                obj.add_css_class("video-widget");
            }

            Webview::instance().set_is_enclosure_fullscreen(is_fullscreen);
            window.set_fullscreened(is_fullscreen);
        }

        #[template_callback]
        fn on_image_error(&self) {
            self.image_overlay.set_visible(false);
        }

        #[template_callback]
        fn on_overlay_play(&self) {
            self.image_overlay.set_visible(false);
            if let Some(player) = self.video.player() {
                player.play();
            }
        }

        fn stop(&self) {
            self.obj().set_can_seek(true);

            if let Some(player) = self.video.player() {
                let position = player.position();

                let is_finished = player
                    .queue()
                    .and_then(|q| q.current_item())
                    .map(|item| item.duration())
                    .map(|duration| position / duration >= constants::VIDEO_END_THRESHOLD_PERCENT)
                    .unwrap_or(false);

                let position = if is_finished { 0.0 } else { position };
                player.stop();

                self.set_enclosure_position(position);
            }
        }

        fn set_enclosure_position(&self, position: f64) {
            let enclosure = self.enclosure.borrow().clone();
            enclosure.set_position(position);

            if let Some(article) = ArticleViewColumn::instance().article() {
                let mut enclosures = article.enclosures();

                for e in enclosures.as_mut() {
                    if e.url() == enclosure.url() {
                        e.set_position(enclosure.position());
                        break;
                    }
                }
            }

            let enclosure = Enclosure::from(enclosure);

            TokioRuntime::instance().spawn(async move {
                if let Some(news_flash) = App::news_flash().read().await.as_ref() {
                    _ = news_flash.update_enclosure(&enclosure);
                }
            });
        }

        fn start_skip_cooldown(&self) {
            GtkUtil::remove_source(self.skip_cooldown_source.take());
            self.skip_seconds
                .set(self.skip_seconds.get() + constants::VIDEO_SEEK_SECONDS);
            self.show_skip_overlay();

            self.skip_cooldown_source.borrow_mut().replace(glib::timeout_add_local(
                Duration::from_millis(constants::VIDEO_SEEK_COOLDOWN_MS),
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    #[upgrade_or_panic]
                    move || {
                        imp.skip_cooldown_source.take();
                        imp.skip();
                        ControlFlow::Break
                    }
                ),
            ));
        }

        fn skip(&self) {
            if let Some(player) = self.video.player() {
                let position = player.position();
                player.seek(position + self.skip_seconds.get());

                self.obj().set_can_seek(false);
                self.skip_seconds.set(0.0);
            }
        }

        fn show_skip_overlay(&self) {
            let message = format!("+{}s", self.skip_seconds.get());
            self.billboard.post_message("media-seek-forward-symbolic", &message);
        }

        fn start_rewind_cooldown(&self) {
            GtkUtil::remove_source(self.rewind_cooldown_source.take());
            self.rewind_seconds
                .set(self.rewind_seconds.get() + constants::VIDEO_SEEK_SECONDS);
            self.show_rewind_overlay();

            self.rewind_cooldown_source
                .borrow_mut()
                .replace(glib::timeout_add_local(
                    Duration::from_millis(constants::VIDEO_SEEK_COOLDOWN_MS),
                    clone!(
                        #[weak(rename_to = imp)]
                        self,
                        #[upgrade_or_panic]
                        move || {
                            imp.rewind_cooldown_source.take();
                            imp.rewind();
                            ControlFlow::Break
                        }
                    ),
                ));
        }

        fn rewind(&self) {
            if let Some(player) = self.video.player() {
                let position = player.position();
                player.seek(position - self.rewind_seconds.get());

                self.obj().set_can_seek(false);
                self.rewind_seconds.set(0.0);
            }
        }

        fn show_rewind_overlay(&self) {
            let message = format!("-{}s", self.rewind_seconds.get());
            self.billboard.post_message("media-seek-backward-symbolic", &message);
        }
    }
}

glib::wrapper! {
    pub struct VideoWidget(ObjectSubclass<imp::VideoWidget>)
        @extends Widget, Bin,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl From<&GEnclosure> for VideoWidget {
    fn from(value: &GEnclosure) -> Self {
        let author = value
            .author()
            .unwrap_or_else(|| value.feed_title().unwrap_or_else(|| i18n("Unkown Author")));
        let title = value
            .title()
            .unwrap_or_else(|| value.article_title().unwrap_or_else(|| i18n("Unknown Enclosure")));
        let item = MediaItem::new(&value.url());

        Object::builder()
            .property("author", author)
            .property("title", title)
            .property("enclosure", value)
            .property("item", item)
            .build()
    }
}
