#define CONFIG_KERNELD
#include <linux/kerneld.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>
#include <linux/unistd.h>
#include <asm/param.h>
#include <signal.h>
#include <errno.h>

/*
 * This is kerneld, the user level handler of kernel requests.
 * Copyright (C) 1995, Bjorn Ekwall <bj0rn@blox.se>
 * See the file "COPYING" for your rights.
 *
 *
 * The requests arrive on a specific IPC message queue,
 * and the type of the message specifies the request.
 *
 * The message consists of a "long id" followed by "char text[]".
 * The parameter for the request is stored in the "text" field.
 * If "id" is non-zero the kernel expects an answer, where the
 * message type of the answer should be "id".
 * In the return message, the "id" will contain the status of the request.
 * If the "id" of the request is zero, the kernel does not expect an answer.
 *
 * The "text" field can be used to return any generated information.
 * (See KERNELD_SYSTEM)
 *
 * (C) Bjorn Ekwall <bj0rn@blox.se> in May 1995
 * This software is placed under the GPL, the text of which can be found
 * in the root directory of this release.
 */

#define MAXCMD 255
/*
 * The _real_ maximum message type is KERNELD_MAXCMD,
 * but we reserve some types for other daemons...
 */

#define MSIZE 1024 /* might be MSGMAX instead... almost 4k nowadays */
#define DELAY_TIME 60 /* adjustable with the "delay=..." parameter */
#define JOB_DONE 0

#ifdef DEBUG
time_t t0;
#define DPRINT(arg) if (debug) { printf("%d:", (int)time(0) - (int)t0);printf arg; }
#else
#define DPRINT(arg)
#endif

int qid;
/*
 * If "kerneld" is called with the argument "keep",
 * all requests for module release will be ignored.
 *
 * If "delay=10" is used as an argument, the default delay
 * will be changed from the preset value of 60.
 */
int keep = 0;
int delay = DELAY_TIME;
int debug = 0;
int want_type = -MAXCMD;
int dev_null;

/*
	Generate an error message with the syslog facility.
*/
void kerneld_error (const char *ctl, ...)
{
	char buf[1000];
	va_list list;
	va_start (list,ctl);
	vsprintf (buf,ctl,list);
	syslog (LOG_ERR,"%s",buf);
}

void kerneld_perror ()
{
	kerneld_error ("error: %s",strerror(errno));
	exit (1);
}

void handle_child(int sig);
struct sigaction child_action = {
	handle_child,
	0,
	SA_RESTART | SA_NOMASK,
};

void handle_timer(int sig);
struct sigaction timer_action = {
	handle_timer,
	0,
	SA_RESTART | SA_NOMASK,
};

struct job {
	struct job *next;
	pid_t pid;
	int status;
	int reply_size;
	int reply_fd;
	struct kerneld_msg msg;
};

struct job *job_head = NULL;
volatile int timer = 0;
extern int errno;

void
handle_child(int sig)
{
	struct job *job;
	int pid;
	int status;

	if ((pid = waitpid(-1, &status, WNOHANG)) <= 0)
		return;

	for (job = job_head; job; job = job->next) {
		if (job->pid == pid) {
			job->pid = JOB_DONE;
			job->status = WEXITSTATUS(status);
			/* don't break, more jobs might be waiting... */
			DPRINT(("SIGCHLD: job (%08lx), pid=%d, status=%d\n",
				(long)job, pid, job->status));
		}
	}
}

void
handle_timer(int sig)
{
	timer = 1;
	alarm(delay);
}

/*
 * Execute the requested kerneld task, return pid or -errno.
 * If pid is set to 0 then no process has been spawned.
 *
 * Note that the job has been allocated with calloc, so all
 * fields are already zero (including the status).
 */
int
spawn_it(struct job *newjob)
{
	struct job *job;
	int pipes[2];
	int status;
	int pid = 0;
	int op;

	switch (op = newjob->msg.mtype) {
	case KERNELD_CANCEL_RELEASE_MODULE:
		if (keep)
			break;
		/* else */
		for (job = job_head; job; job = job->next) {
			if ((job->msg.mtype == KERNELD_DELAYED_RELEASE_MODULE)
			     && (job->pid != JOB_DONE) &&
			     (strcmp(job->msg.text, newjob->msg.text) == 0)) {
				kill(job->pid, SIGINT);
				DPRINT(("job (%08lx), pid %d terminated\n",
					(long)job, job->pid));
			}
		}
		break;
	
	case KERNELD_SYSTEM:
		if (newjob->msg.id) /* reply wanted */
			pipe(pipes);
		if ((pid = fork()) == 0) {
			if (newjob->msg.id) { /* reply wanted */
				close(1);
				dup(pipes[1]);
				close(pipes[0]);
				close(pipes[1]);

				close(0);close(2);
				dup(dev_null); dup(dev_null);
			}
			else {
				close(0);close(1);close(2);
				dup(dev_null); dup(dev_null); dup(dev_null);
			}

			/* signal(SIGCHLD, SIG_DFL); */
			status = system(newjob->msg.text);
			exit(WEXITSTATUS(status));
		}
		if ((pid > 0) && newjob->msg.id) { /* reply wanted */
			newjob->reply_fd = pipes[0];
			close(pipes[1]);
		}
		break;

	case KERNELD_REQUEST_MODULE:
		if (!keep)
			/*
			 * re-arm the timer for auto-removal,
			 * keep an auto-loaded module at least this long
			 */
			alarm(delay);

		if ((pid = fork()) == 0) {
			close(0);close(1);close(2);
			dup(dev_null); dup(dev_null); dup(dev_null);
			execlp("/sbin/modprobe", "modprobe",
#ifndef NO_EXTRA_OPTS
						"-k", "-s",
#endif
						newjob->msg.text, 0);
			exit(1);
		}
		break;
		
	case KERNELD_DELAYED_RELEASE_MODULE:
	case KERNELD_RELEASE_MODULE:
		if (keep)
			pid = 0;
		/* else */
		if ((pid = fork()) == 0) {
			close(0);close(1);close(2);
			dup(dev_null); dup(dev_null); dup(dev_null);
			if (op == KERNELD_DELAYED_RELEASE_MODULE)
				sleep(delay);
			execlp("/sbin/modprobe", "modprobe", "-r",
#ifndef NO_EXTRA_OPTS
						 "-s",
#endif
						newjob->msg.text, 0);
			exit(1);
		}
		break;

	case KERNELD_REQUEST_ROUTE:
		/* check if there is a previous similar route request running */
		for (job = job_head; job; job = job->next) {
			if ((job->msg.mtype == KERNELD_REQUEST_ROUTE)
			     && job->pid && (job->pid != JOB_DONE) &&
			     (strcmp(job->msg.text, newjob->msg.text) == 0)) {
				/* wait for the same pid */
				pid = job->pid;
				DPRINT(("job (%08lx), depends on job(%08lx)\n",
					(long)newjob, (long)job));
				break; /* for-loop */
			}
		}
		/* no previous request is running, so start a new one */
		if ((pid == 0) && ((pid = fork()) == 0)) {
			close(0);close(1);close(2);
			dup(dev_null); dup(dev_null); dup(dev_null);
			execlp("/sbin/request-route", "request-route",
				newjob->msg.text, 0);
			exit(1);
		}
		break;
		
	case MAXCMD: /* debug entry, used by kdstat */
		{
			char *p = newjob->msg.text;
			struct msgbuf msgp;
			int done = 0;
			int count = 0;

			if (strcmp(p, "debug") == 0)
				debug = 1;
			else if (strcmp(p, "nodebug") == 0)
				debug = 0;
			else if (strcmp(p, "keep") == 0)
				keep = 1;
			else if (strcmp(p, "nokeep") == 0)
				keep = 0;
			else if (strncmp(p, "delay=", 6) == 0) {
				count = atoi(p + 6);
				if (count) {
					delay = count;
					alarm(delay);
				}
			}
			else if (strcmp(p, "flush") == 0) {
				count = 0;
				while (msgrcv(qid, &msgp, 1, 0,
					IPC_NOWAIT | MSG_NOERROR) >= 0)
					++count;
				sprintf(p, "flushed %d entries\n", count);
				p += strlen(p);

			}

			sprintf(p, "pid=%d, delay=%d, %skeep, %sdebug\n",
				getpid(), delay, keep?"":"no", debug?"":"no");
			p += strlen(p);

			count = 0;
			for (job = job_head; job; job = job->next) {
				if (p - newjob->msg.text > MSIZE - 40) {
					done = 1;
					break; /* for-loop */
				}
				++count;
				if (count == 1) {
					sprintf(p, "job queue:\n");
					p += strlen(p);
				}
				sprintf(p, "pid %d, type %d ('%s')\n",
				job->pid, (int)job->msg.mtype, job->msg.text);
				p += strlen(p);
			}

			if (!done && !count) {
				sprintf(p, "no jobs waiting\n");
				p += strlen(p);
			}

		}
		newjob->reply_size = strlen(newjob->msg.text);
		break;
#if 0
	case GENERIC_EXAMPLE:
		/* get the parameter from newjob->msg.text */
		/* if it's illegal, set newjob->status to the error number */
		/* else if it is an internal job: */
		/*     do it and copy any result back to newjob->msg.text */
		/*     update newjob->reply_size with the size (in bytes) */
		/*     update newjob->status if it shouldn't be 0 */
		/* else the job should be spawned: */
		/*     set up the parameters and fork a new process */
		/*     set pid to the pid of the new process */
		break;
#endif

	default: /* unknown */
		kerneld_error ("kerneld: unknown message type: %d", op);
		newjob->status = 1; /* failed... ? */
		break;
	}

	return pid;
}

/*
 * Look for finished jobs and take care of them.
 * Return < 0 if fatal error
 */
int
finish_jobs()
{
	struct job *job;
	struct job **pjob;
	int status = 0;

	DPRINT(("finish_jobs\n"));
	for (pjob = &job_head, job = job_head; job && (status >= 0); job = *pjob) {
		if (job->pid == JOB_DONE) {
			/*
			 * Does the kernel expect an answer?
			 */
			if (job->msg.id) {
				if (job->reply_fd) {
					struct timeval timeout = { 0, 0 };
					fd_set	readfds;

					FD_ZERO(&readfds);
					FD_SET(job->reply_fd, &readfds);
					if ((select(job->reply_fd + 1, &readfds,
						(fd_set *)NULL,
						(fd_set *)NULL, &timeout) > 0) &&
				    	FD_ISSET(job->reply_fd, &readfds)) {
						job->reply_size = read(job->reply_fd,
							job->msg.text, MSIZE);
					}
					close(job->reply_fd);
				}
				job->msg.mtype = job->msg.id;
				job->msg.id = job->status;

				status = msgsnd(qid, (struct msgbuf *)&job->msg,
						((job->reply_size > 0) ?
						job->reply_size : 0)
						+ sizeof(long), 0);
			}
			/*
			 * unlink the job
			 */
			DPRINT(("job (%08lx) unlinked\n", (long)job));
			*pjob = job->next;
			free(job);
		}
		else /* job not done */
			pjob = &(job->next);
	}

	return status;
}

void
do_timer()
{
	extern int syscall(int, ...);

	timer = 0;
	syscall( __NR_delete_module, NULL);
}

int
main(int argc, char *argv[])
{
	struct job *job;
	int pid;
	int status;

	/* make me a daemon */
	if (fork() != 0)
		exit(0);

	openlog ("kerneld",0,LOG_DAEMON);
	if ((qid = msgget(IPC_PRIVATE, S_IRWXU | IPC_CREAT | IPC_KERNELD)) < 0){
		kerneld_error("Can't initialise message queue: %s",strerror(errno));
		exit(1);
	}

#ifdef DEBUG
	t0 = time(0);
#endif

	setbuf(stdout, NULL);
	syslog (LOG_INFO,"started, pid=%d, qid=%d", getpid(), qid);

	setsid();

	if (sigaction(SIGCHLD, &child_action, NULL) < 0) {
		kerneld_perror();
	}
	if (sigaction(SIGALRM, &timer_action, NULL) < 0) {
		kerneld_perror();
	}

	while (argc > 1) {
		if (strcmp(argv[1], "keep") == 0)
			keep = 1;
		else if (strncmp(argv[1], "delay=", 6) == 0)
			delay = atoi(&(argv[1][6]));
		else if (strcmp(argv[1], "debug") == 0)
			debug = 1;
		else if (strncmp(argv[1], "type=", 5) == 0)
			want_type = atoi(&(argv[1][5]));
		++argv;
		--argc;
	}

	dev_null = open("/dev/null", O_RDWR);

	if (!keep)
		alarm(delay); /* start the timer task */

	for (;;) {
		job = (struct job *)calloc(1, sizeof(struct job) + MSIZE);
		if (job == NULL) {
			kerneld_perror();
		}

	restart:
		while ((status = msgrcv(qid, (struct msgbuf *)&job->msg,
			sizeof(long) + MSIZE, want_type, MSG_NOERROR)) == 0)
			/*keep on*/;

		if (status < 0) {
			/* was the msgrcv interrupted by a signal? */
			if (errno == EINTR) {
				if (timer) {
					do_timer();
				}
				if (finish_jobs() < 0) {
					break; /* fatal error */
				}
				/* else */
				goto restart; /* wait for message */
			}
			/* else */
			break; /* some other error */
		}
		/* else, we got a message */

		/* null-terminate the message */
		status = (status >= MSIZE) ? (MSIZE - 1) : status;
		((struct msgbuf *)&job->msg)->mtext[status] = '\0';

		DPRINT(("job (%08lx) scheduled, type %ld\n", (long)job, job->msg.mtype));

		if ((pid = spawn_it(job)) < 0)
			break; /* fatal error */

		/*
		 * Link this job into the job queue if something was
		 * spawned or if someone is waiting for an answer.
		 */
		if ((pid > 0) || (job->msg.id != 0)) {
			job->pid = pid;
			job->next = job_head;
			job_head = job;
			DPRINT(("job (%08lx) stored, pid=%d\n", (long)job,pid));
			/* take care of all tasks, including non-spawned ones */
			if (finish_jobs() < 0)
				break; /* fatal error */
		}
		else { /* all work requested has been performed */
			DPRINT(("job (%08lx) released\n", (long)job));
			free(job);
		}
	}

	kerneld_perror();
	return 1;
}
