// == Globals and Constants
var gTreeView;
var gFilterBox;
var gFilterLabel;
var gTree;
var gDeck;
var gFetchInProgress;
var gRcCmd;
var gMainBox;

var gFilterHistory = [];

var NS_HRC = "@merrillpress.com/2008/hrc";
var NS_XHTML = "http://www.w3.org/1999/xhtml";
var NS_XUL = "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul";

var COV_MEDIOCRE = 50; /* >= */
var COV_GOOD     = 75; /* >= */
var COV_GREAT    = 95; /* >= */

var PAGE_INITIAL    = 0;
var PAGE_RECORDING  = 1;
var PAGE_PROCESSING = 2;
var PAGE_RESULTS    = 3;

// Number of lines from the end of the file to look for
// file-local varirevertables
var LOCAL_VARS_LINES = 200;

var RE_STRING_LITERAL = '"(?:[^\"\\\\]|\\\\.)*"';
var RE_NUMBER = '-?[0-9]+(?:\\.[0-9]+)?';
var RE_SYMBOL = '[a-zA-Z-]+';

// If this regexp matches, eval is safe, and we're looking at a string
// or a number literal
var RE_SAFE_EVAL = new RegExp('^\\s*(?:' + RE_STRING_LITERAL + '|' +
                              RE_NUMBER + ')\\s*$');

// == Functions
// === State Transitions
function trans_results_to_recording() {
  assert(function() (gDeck.selectedIndex == PAGE_RESULTS ||
                     gDeck.selectedIndex == PAGE_INITIAL ));

  window.setCursor('wait');
  pump_events();
  
  gDeck.selectedIndex = PAGE_RECORDING;
  force_redraw();

  gHrc.clearData();
  gHrc.start();
  gRcCmd.setAttribute('label', 'Stop Recording');
  gFilterBox.setAttribute('disabled', 'true');
  gFilterLabel.setAttribute('disabled', 'true');

  window.setCursor('auto');
}

function trans_recording_to_processing() {
  assert(function() gDeck.selectedIndex == PAGE_RECORDING);
  assert(function() !gFetchInProgress);

  gDeck.selectedIndex = PAGE_PROCESSING;
  force_redraw();
  
  gHrc.stop();
  gRcCmd.setAttribute('label', 'Record Coverage');
  gRcCmd.setAttribute('disabled', 'true');
  window.setCursor('wait');
  pump_events();

  var raw_results = gHrc.getData({});
  var sources_to_get = raw_results.map(function(result) result.uri);

  function on_fetch_done(raw_sources) {
    trans_processing_to_results(raw_results, raw_sources);
  }

  gFetchInProgress = new ASyncURLFetcher(
    sources_to_get, on_fetch_done);
}

// Parse a line like "-*- var1: value1; var2: value2 -*-" and put the
// result in vars. Don't evaluate the variables.
function parse_variable_headline(vars, line) {
  var m = /^.*?-\*- (.*-\*-).*?$/.exec(line);
  if(m) {
    var content = m;
    var var_re = new RegExp('\\s*([a-zA-Z-]+)\\s*:\\s*' +
                            '(' + RE_STRING_LITERAL +
                            '|' + RE_NUMBER +
                            '|' + RE_SYMBOL +
                            // terminated by ';' or '; -*-' or just '-*-'
                            ')\\s*(?:;|;?\\s*-\\*-)',
                            'g');

    while((m = var_re.exec(content))) {
      vars[m[1]] = m[2];
    }
  }
}

// Parse emacs-style local variables at the end of a file. tag is the
// local-variables tag. For Emacs, it'd be 'Local Variables'.
function parse_variable_footer(vars, lines, tag) {
  var start = Math.max(0, lines.length - LOCAL_VARS_LINES);
  var m, var_line_regexp = null;
  var continued_regexp = null;
  var continued_var = null;

  var tag_regexp = new RegExp('^(.*)' + quote_regexp(tag) + ':(.*)$');
  
  for(var i = start; i < lines.length; ++i) {
    var line = lines[i];
    
    if(continued_var) {
      if((m = continued_regexp.exec(line))) {
        var cv = m[1];
        if(cv.length && cv[cv.length - 1] === '\\') {
          cv = cv.substring(0, cv.length - 1);
        }

        vars[continued_var] += cv;

        if(m[1].length && m[1][m[1].length - 1] !== '\\') {
          continued_var = null;
        }
      } else {
        continued_var = null; // syntax error; ignore?
      }
    } else if((m = tag_regexp.exec(line))) {
      var_line_regexp = new RegExp(
        '^' + quote_regexp(m[1]) + '([a-zA-Z-]+):(.*)' +
          quote_regexp(m[2]) + '$');
      continued_regexp = new RegExp(
        '^' + quote_regexp(m[1]) + '(.*)' + quote_regexp(m[2]) + '$');
      continued_var = null;
    } else if(var_line_regexp && (m = var_line_regexp.exec(line))) {
      var vn = m[1];
      var vv = m[2];

      if(vv.length && vv[vv.length - 1] === '\\') {
        vv = vv.substring(0, vv.length - 1);
        continued_var = vn;
      }

      vars[vn] = vv;

      if(vn === 'End') {
        var_line_regexp = null;
        continued_regexp = false;
        continued_var = null;
      }
    }
  }
}

/**
  * Parse the Emacs-style local variables of the given file (made up
  * of lines.) Only string and number variables are returned.
  *
  * @param lines: Array of strings, one for each line
  * @return: Object giving variable values
  */
function get_local_variables(lines) {
  var vars = {};
  if(lines.length > 0) {
    parse_variable_headline(vars, lines[0]);
  }

  if(lines.length > 1) {
    parse_variable_headline(vars, lines[1]);
  }

  parse_variable_footer(vars, lines, 'Local Variables');
  parse_variable_footer(vars, lines, 'Moz Local Variables');

  var parsed_vars = {};

  for(var vn in vars) {
    var vv = vars[vn];
    if(RE_SAFE_EVAL.test(vv)) {
      parsed_vars[vn] = eval(vv);
    } else {
      // if the variable is not safe to eval, just strip whitespace
      vv = vv.replace(/^\s*(.*)\s*$/, '$1');
      if(vv === 'null' || vv === 'nil') {
        vv = null;
      }
      
      parsed_vars[vn] = vv;
    }
  }

  return parsed_vars;
}

/**
  * Given a string containing an Emacs regular expression, return an
  * equivalent Javascript regular expression as a string.
  *
  * Fortunately, Javascript's regular expression functionality is a
  * superset of Emacs's, even if the syntax is different.
  *
  */
function translate_emacs_regexp(regexp) {
  var len = regexp.length;
  var res = [];

  // Simple state machine
  
  var STATE_NORMAL                   = 0;
  var STATE_ESCAPED                  = 1;
  var STATE_SYNTAX                   = 2;
  var STATE_NOT_SYNTAX               = 3;
  var STATE_SYMBOL_WORD              = 4;
  var STATE_ALTERNATIVE_INITIAL      = 5;
  var STATE_ALTERNATIVE              = 6;
  var STATE_ALTERNATIVE_AFTER_OB     = 7;
  var STATE_ALTERNATIVE_AFTER_COLON1 = 8;
  var STATE_ALTERNATIVE_AFTER_COLON2 = 9;
  
  var state              = STATE_NORMAL;
  var current_char_class = null;

  // Emacs syntax for Javascript. They're hardcoded of course, since
  // we don't have real syntax tables. These are approximate, of
  // course, but should work all right for our purposes.
  var syntax = {
    ' ': '\\s',
    '-': '\\s',
    'w': '[a-zA-Z]',
    '_': '(?:\\w|\\$)',
    '.': '[[](){}%&*+/<>\\\\|#\'\",.:;^-]',
    '(': '\\(',
    ')': '\\(',
    '"': '["\']',
    '\\': '\\',
    '$': '(?!.$)', // match nothing, since JS has no $ syntax
    '<': /* start of comment*/ '/(?=/)',
    '>': '\n'
  };

  // Again, approximate, but should work. These must all be valid
  // inside a JS character alternative
  var classes = {
    'alnum': 'a-zA-Z0-9',
    'alpha': 'a-zA-Z',
    'blank': ' \t',
    'digit': '0-9',
    'lower': 'a-z',
    'punct': '(){}%&*+/<>\\\\|#\'\",.:;^-',
    'space': ' \t\f\n\r',
    'upper': 'A-Z',
    'word': 'a-zA-Z0-9',
    'xdigit': '0-9a-fA-F'
  };

  for(var i = 0; i < len; ++i) {
    var c = regexp[i];

    if(state === STATE_NORMAL)
    {
      if(c === '\\') {
        state = STATE_ESCAPED;
      } else if(c === '(' || c === ')'
                || c === '|' || c === '{' || c === '}') {
        // these are special in JS, but not Emacs, so escape them
        res.push('\\' + c);
      } else if(c === '[') {
        res.push('[');
        state = STATE_ALTERNATIVE_INITIAL;
      } else {
        // ., *, +, ?, ^, and $ mean the same in JS as they do in
        // Emacs, and can go in unchanged
        res.push(c);
      }
    }
    else if(state === STATE_ESCAPED)
    {
      if(c === '|' || c === '{' || c === '}' ||
         c === '(' || c === '(') {
        // these mean the same thing in JS, but they're unescaped
        // there and escaped in Emacs
        res.push(c);
        state = STATE_NORMAL;
      } else if(c === '1' || c === '2' || c === '3' ||
                c === '4' || c === '5' || c === '6' ||
                c === '7' || c === '8' || c === '9' ||
                c === 'w' || c === 'W' || c === 'b' ||
                c === '.' || c === '*' || c === '+' ||
                c === '?' || c === '[' || c === ']' ||
                c === '^' || c === '$' || c === '\\') {
        // these escapes are (basically) the same in JS and Emacs
        res.push('\\' + c);
        state = STATE_NORMAL;
      } else if(c === 's') {
        state = STATE_SYNTAX;
      } else if(c === 'S') {
        state = STATE_NOT_SYNTAX;
      } else if(c === 'c' || c === 'C') {
        throw new Error('character categories not supported');
      } else if(c === '`') {
        res.push('^');
        state = STATE_NORMAL;
      } else if(c === '\'') {
        res.push('$');
        state = STATE_NORMAL;
      } else if(c === 'B') {
        res.push('(?!\\b)');
        state = STATE_NORMAL;
      } else if(c === '<') { // beginning of a word
        res.push('\\b(?=\\w)');
        state = STATE_NORMAL;
      } else if(c === '>') { // end of a word
        res.push('\\b(?=\\W|$)');
        state = STATE_NORMAL;
      } else if(c === '_') {
        state = STATE_SYMBOL_WORD;
      } else {
        throw new Error('invalid escape: "' + c + '"');
      }
    }
    else if(state === STATE_ALTERNATIVE_INITIAL)
    {
      if(c === '[') {
        state = STATE_ALTERNATIVE_AFTER_OB;
      } else if(c === '\\') {
        res.push('\\\\');
        state = STATE_ALTERNATIVE;
      } else if(c === '^') {
        res.push(c);
        state = STATE_ALTERNATIVE_INITIAL;
      } else {
        res.push(c);
        state = STATE_ALTERNATIVE;
      }
    }
    else if(state === STATE_ALTERNATIVE) {
      if(c === ']') {
        state = STATE_NORMAL;
        res.push(']');
      } else if(c === '\\') {
        res.push('\\\\');
        state = STATE_ALTERNATIVE;
      } else if(c === '[') {
        state = STATE_ALTERNATIVE_AFTER_OB;
      } else {
        res.push(c);
        state = STATE_ALTERNATIVE;
      }
    }
    else if(state === STATE_ALTERNATIVE_AFTER_OB)
    {
      if(c === ':') { // start of character class
        current_char_class = '';
        state = STATE_ALTERNATIVE_AFTER_COLON1;
      } else if(c === ']') {
        res.push('[]');
        state = STATE_NORMAL;
      } else {
        res.push('[' + c);
        state = STATE_ALTERNATIVE;
      }
    }
    else if(state === STATE_ALTERNATIVE_AFTER_COLON1) {
      if(c === ':') {
        if(! (current_char_class in classes)) {
          throw new Error('unknown character class: "' +
                          current_char_class + '"');
        }
        
        res.push(classes[current_char_class]);
        state = STATE_ALTERNATIVE_AFTER_COLON2;
      } else {
        current_char_class += c;
        state = STATE_ALTERNATIVE_AFTER_COLON1;
      }
    }
    else if(state === STATE_ALTERNATIVE_AFTER_COLON2) {
      if(c === ']') {
        state = STATE_ALTERNATIVE;
      } else {
        throw new Error("expected ']' after second ':'");
      }
    }
    else if(state === STATE_SYNTAX)
    {
      if(! (c in syntax)) {
        throw new Error('invalid syntax class:' + c);
      }

      res.push(syntax[c]);
      state = STATE_NORMAL;
    }
    else if(state === STATE_NOT_SYNTAX)
    {
      if(! (c in syntax)) {
        throw new Error('invalid syntax class:' + c);
      }

      res.push('(?!' + syntax[c] + ').');
      state = STATE_NORMAL;
    }
    else if(state === STATE_SYMBOL_WORD)
    {
      if(c === '<') { // match start of symbol
        res.push('\\b(?=\\w)(?!_)');
      } else if(c === '>') { // match end of symbol
        res.push('\\b(?=\\W|$)');
      } else {
        throw new Error('invalid \\_X construct');
      }

      state = STATE_NORMAL;
    }
  }

  if(state !== STATE_NORMAL) {
    throw new Error('unterminated regular expression');
  }

  return res.join('');
}

var gBitset = Cc['@merrillpress.com/hrbitset;1'];
var gIHRBitset = Ci.IHRBitset;
function make_bitset() {
  return gBitset.createInstance(gIHRBitset);
}

function ParseEntry(parent, depth, start) {
  assert(function() parent === null || parent.depth < depth);
  this.parent = parent;
  
  this.depth = depth;
  this.start = start;
  this.end = null;
  this.children = [];

  this.sourceLines = null;
  if(parent) {
    this.sourceLines = parent.sourceLines;
  }
}

ParseEntry.prototype = {
  constructor: ParseEntry,

  addChild: function(child) {
    this.children.push(child);
  },

  setEnd: function(max_line) {
    this.end = max_line;
  },

  getSourceLine: function() {
    if(this.sourceLines && this.start < this.sourceLines.length) {
      return this.sourceLines[this.start - 1];
    }

    return null;
  },

  finish: function(raw_result, level) {
    assert(function() this.end !== null);
    
    if(this.parent) {
      var mask = this.mask = make_bitset();
      mask.setBits(this.start, this.end);
      
      this.lines = this.parent.lines.clone();
      this.lines.and(mask);
      this.unExecLines = this.parent.unExecLines.clone();
      this.unExecLines.and(mask);
    } else {
      var e = raw_result.executedLines;
      var p = raw_result.potentialLines;
      e.matchSize(p);
      var u = p.clone();
      var not_e = e.clone(); not_e.flip();
      u.and(not_e);

      this.lines = p;
      this.unExecLines = u;
    }

    if(level === undefined) {
      level = 0;
    }
    
    this.level = level;
    this.open = true;
    this.uri = raw_result.uri;
    this.nrUnExecLines = this.unExecLines.numberBitsSet;
    this.nrLines = this.lines.numberBitsSet;
    this.nrExecLines = this.nrLines - this.nrUnExecLines;
    this.coveragePercent = 100 * (this.nrLines - this.nrUnExecLines) /
      this.nrLines;

    for(var i = 0; i < this.children.length; ++i) {
      this.children[i].finish(raw_result, level + 1);
    }
  }
};

/**
  * Parse Emacs outline information. Return the ParseEntry
  * representing the whole file.
  */
function parse_outline(source_lines, raw_result) {
  var entry = new ParseEntry(null, 0, 1);
  entry.sourceLines = source_lines;

  if(source_lines) {
    var max_line = source_lines.length;
    var vars = get_local_variables(source_lines);

    var outline_regexp = null;
    if('outline-regexp' in vars) {
      try {
        outline_regexp = new RegExp(translate_emacs_regexp(vars['outline-regexp']));
      } catch(ex) {
        dump(ex);
      }
    }

    if(outline_regexp) {
      var lineno = 1;
      for(; lineno <= source_lines.length; ++lineno) {
        var line = source_lines[lineno - 1];
        var m = outline_regexp.exec(line);

        if(m) {
          var depth = m[0].length;
          dump('NEW (' + entry.depth + ' -> ' + depth + '):' + line + '\n');
          
          while(depth <= entry.depth) {
            dump('CLOSING (' + entry.depth + ' -> ' +
                 entry.parent.depth + '):' +
                 source_lines[entry.start - 1] + '\n');
            entry.setEnd(lineno);
            entry = entry.parent;
            assert(function() entry);
          }

          var new_entry = new ParseEntry(entry, depth, lineno);
          entry.addChild(new_entry);
          entry = new_entry;
        }
      }
      
      while(entry.parent) {
        entry.setEnd(lineno);
        entry = entry.parent;
      }
    }
  } else {
    var max_line =
      raw_result.potentialLines.getHighestBit();
    if(max_line > 0) {
      max_line--;
    }
  }

  entry.setEnd(max_line);
  entry.finish(raw_result);
  return entry;
}

function trans_processing_to_results(raw_results, raw_sources) {
  assert(function() gDeck.selectedIndex == PAGE_PROCESSING);
  assert(function() gFetchInProgress);
  
  gFetchInProgress = null;
  var results = [];
  
  for(var i = 0; i < raw_results.length; ++i) {
    var raw_result = raw_results[i];
    var uri = raw_result.uri;

    if(raw_sources[uri]) {
      var source_lines = raw_sources[uri].split(/\r?\n/);
    } else {
      var source_lines = null;
    }

    results.push(parse_outline(source_lines, raw_result));
    pump_events();
  }
    
  gRcCmd.removeAttribute('disabled');
  gFilterBox.removeAttribute('disabled');
  gFilterLabel.removeAttribute('disabled');
  gTreeView.update(results);
  window.setCursor('auto');
  gDeck.selectedIndex = PAGE_RESULTS;
}

// === Other

function save_filter(f) {
  var pref_svc = Cc["@mozilla.org/preferences-service;1"]
    .getService(Ci.nsIPrefService);
  var branch = pref_svc.getBranch(null);
  branch.setCharPref('hrtimer.hrcov.filter', f);
}

function get_saved_filter() {
  var pref_svc = Cc["@mozilla.org/preferences-service;1"]
    .getService(Ci.nsIPrefService);
  var branch = pref_svc.getBranch(null);
  try {
    return branch.getCharPref('hrtimer.hrcov.filter');
  } catch(ex) {
    var f = '^https?://';
    save_filter(f);
    return f;
  }
}

function on_filter_reverted() {
  gFilterBox.value = gFilterString;
  dump('reverted to:' + gFilterString + '\n');
  on_filter_input();
}

function on_filter_change() {
  gFilterString = gFilterBox.value;
}

function on_filter_input() {
  var regexp_s = gFilterBox.value;

  if(regexp_s === '') {
    regexp_s = '.*';
  }

  try {
    var regexp = new RegExp(regexp_s);
  } catch(ex) {
    gFilterBox.setAttribute('class', 'regexpError');
    return;
  }

  gFilterBox.setAttribute('class', '');
  gTreeView.setFilter(regexp);
  save_filter(gFilterBox.value);
}

function view_source(url, lineno_start, lineno_end) {
  openDialog("chrome://global/content/viewSource.xul",
             "_blank",
             "all,dialog=no",
             url, null, null, lineno_start, false);
}

function sort_data(data, sortby) {
  var key = sortby.substring(4); // strip off leading col_
  if(key === 'coverage') {
    key = 'coveragePercent';
  }

  function sorter(a, b) {
    var val1 = a[key];
    var val2 = b[key];

    if(val1 < val2) {
      return -1;
    }

    if(val2 < val1) {
      return 1;
    }

    return 0;
  }

  data.sort(sorter);
}

function copy_list(aList) {
  return aList.map(function(x) x);
}

/**
  * Calls func for each selected row in a treeview. The argument is
  * the row number.
  */
function for_selected_rows(func) {
  var selection = gTree.view.selection;
  var num_ranges = selection.getRangeCount();
  var min = {};
  var max = {};

  for(var rangeno = 0; rangeno < num_ranges; ++rangeno) {
    selection.getRangeAt(rangeno, min, max);
    for(var rowno = min.value; rowno <= max.value; ++rowno) {
      func(rowno);
    }
  }
}

function on_select() {
  var details = document.getElementById('details');
  var selection = window.getSelection();
  clear_children(details);
  for_selected_rows(function(rowno) {
    var detail = add_detail_xml(
      gTreeView._data[rowno], details);
  });
}

function ranges_from_bitset(bs) {
  var ranges = [];
  var open = null;
  var size = bs.size;

  for(var i = 0; i < size; ++i) {
    if(bs.testBit(i)) {
      if(open === null) {
        open = i;
      }
    } else {
      if(open !== null) {
        ranges.push([open, i - 1]);
        open = null;
      }
    }
  }

  if(open !== null) {
    ranges.push([open, size - 1]);
  }

  return ranges;
}

function on_source_range_clicked(e) {
  e.preventDefault();

  var row = e.target.row;
  var nc_range = e.target.nc_range;
  view_source(row.uri, nc_range[0], nc_range[1]);
}

function on_sourcett_showing(e) {
  var row = document.tooltipNode.row;
  var nc_range = document.tooltipNode.nc_range;

  var f = this.firstChild;
  clear_children(f);

  var rs = row.sourceLines;

  if(rs && rs.length) {
    var first = Math.max(1, nc_range[0] - 3);
    var last = Math.min(nc_range[1] + 3, rs.length);

    var max_digits = (last + '').length + 1;

    for(var i = first; i <= last; ++i) {
      var lineno_string = i + ':';
      while(lineno_string.length < max_digits) {
        lineno_string = ' ' + lineno_string;
      }

      var p = f;

      if(i >= nc_range[0] && i <= nc_range[1]) {
        p = document.createElementNS(NS_XHTML, 'span');
        p.className = 'srcinregion';
        f.appendChild(p);
      }

      p.appendChild(document.createTextNode(lineno_string + rs[i-1] + '\n'));
    }
  } else {
    f.appendChild(document.createTextNode('[source unavailable]'));
  }
}

function add_detail_xml(row, parent) {
  var detail = document.createElementNS(NS_HRC, 'detail');

  var uri = document.createElementNS(NS_HRC, 'uri');
  uri.appendChild(document.createTextNode(row.uri));
  if(row.depth > 0) {
    var sl = row.getSourceLine();
    if(sl) {
      uri.appendChild(document.createTextNode(
        ' ' + sl));
    }
  }
  
  detail.appendChild(uri);

  function add(key_name) {
    var kvpair = document.createElementNS(NS_HRC, 'kvpair');
    var key = document.createElementNS(NS_HRC, 'key');
    key.appendChild(document.createTextNode(key_name));
    var value = document.createElementNS(NS_HRC, 'value');
    kvpair.appendChild(key);
    kvpair.appendChild(value);
    detail.appendChild(kvpair);
    return value;
  }
  
  if(row.nrUnExecLines) {
    var nc = add('Lines not run: ');
    var nc_ranges = ranges_from_bitset(row.unExecLines);
    
    for(var i = 0;  i < nc_ranges.length; ++i) {
      if(i !== 0) {
        nc.appendChild(document.createTextNode(', '));
      }

      var nc_range = nc_ranges[i];
      if(nc_range[0] === nc_range[1]) {
        var nc_range_text = nc_range[0];
      } else {
        var nc_range_text = nc_range[0] + '\u2013' + nc_range[1];
      }

      link = document.createElementNS(NS_XHTML, 'a');
      link.href = '#';
      link.row = row;
      link.nc_range = nc_range;
      link.addEventListener('click', on_source_range_clicked, false);
      link.setAttribute('tooltip', 'sourcett');
      link.appendChild(document.createTextNode(nc_range_text));
      nc.appendChild(link);
    }
  }

  var x = add('Source range: ');
  x.appendChild(document.createTextNode(
    row.start + '\u2013' + row.end));
  
  parent.appendChild(detail);
  return detail;
}

// == Treeview

function HRCovTreeView() {
  this._rawData = this._data = [];
  this.rowCount = 0;
  this._filter = /.*/;
}

HRCovTreeView.prototype = {
  constructor: HRCovTreeView,

  _getTree: function() {
    return this._treebox.treeBody.parentNode;
  },

  _doRefresh: function() {
    var tree = this._getTree();
    var results = [];
    
    for(var i = 0; i < this._rawData.length; ++i) {
      var item = this._rawData[i];
      if(this._filter.test(item.uri)) {
        results.push(item);
      }
    }

    var [sort_resource, sort_direction] =
      update_tree_sorting(tree);
    
    sort_data(results, sort_resource);
    
    if(sort_direction !== 'ascending') {
      results.reverse();
    }

    var expanded_results = [];

    function add_to_expanded(item) {
      expanded_results.push(item);

      if(item.open) {
        var children = item.children;
        var n = children.length;
        for(var j = 0; j < n; ++j) {
          add_to_expanded(children[j]);
        }
      }
    }

    for(var i = 0; i < results.length; ++i) {
      add_to_expanded(results[i]);
    }

    var selection = tree.view.selection;

    var selected_items = [];
    var current_item = null;
    
    var data = this._data;
    
    for_selected_rows(function(row) {
      selected_items.push(data[row]);
    });
    
    delete data;

    if(selection.currentIndex >= 0) {
      current_item = this._data[selection.currentIndex];
    }

    selection.selectEventsSuppressed = true;
    selection.clearSelection();
    
    this._data = expanded_results;
    var old_row_count = this.rowCount;
    this.rowCount = this._data.length;
    this._treebox.rowCountChanged(0, this.rowCount - old_row_count);

    var current_index = null;

    for(var i = 0; i < expanded_results.length; ++i) {
      var item = expanded_results[i];
      
      if(item === current_item) {
        current_index = i;
      }

      for(var j = 0; j < selected_items.length; ++j) {
        if(item === selected_items[j]) {
          selection.rangedSelect(i, i, true);
          break;
        }
      }
    }

    if(current_index !== null) {
      selection.currentIndex = current_index;
    }
    
    selection.selectEventsSuppressed = false;
  },

  setFilter: function(regexp) {
    this._filter = regexp;
    this._doRefresh();
  },

  /* Update with new data, preserving selection */
  update: function(data) {
    this._rawData = data;
    this._doRefresh();
  },

  getProgressMode: function(row, column) {
    return Ci.nsITreeView.PROGRESS_NORMAL;
  },

  getCellText: function(row, column) {
    var id = column.id;
    if(id === 'col_uri') {
      var data = this._data[row];
      if(data.level > 0) {
        return data.sourceLines[data.start - 1];
      }
      
      return data.uri;
    } else if(id === 'col_coveragePercent') {
      return this._data[row].coveragePercent.toPrecision(3)
        + '%';
    } else if(id === 'col_nrExecLines') {
      return this._data[row].nrExecLines;
    } else if(id === 'col_nrUnExecLines') {
      return this._data[row].nrUnExecLines;
    } else if(id === 'col_nrLines') {
      return this._data[row].nrLines;
    } else if(id === 'col_start') {
      return this._data[row].start;
    } else if(id === 'col_end') {
      return this._data[row].end;
    }
  },

  getCellValue: function(row, column) {
    return Math.round(this._data[row].coveragePercent);
  },

  cycleHeader: function(column) {
    var tree = this._treebox.treeBody.parentNode;
    
    if(tree.getAttribute('sortResource') === column.id) {
      if(tree.getAttribute('sortDirection') === 'ascending') {
        tree.setAttribute('sortDirection', 'descending');
      } else {
        tree.setAttribute('sortDirection', 'ascending');
      }
    } else {
      tree.setAttribute('sortResource', column.id);
      tree.setAttribute('sortDirection', 'ascending');
    }

    this._doRefresh();
  },

  setTree: function(treebox) {
    this._treebox = treebox;
  },

  getLevel: function(row) {
    return this._data[row].level;
  },

  hasNextSibling: function(row, afterindex) {
    var data = this._data;
    var this_level = data[row].level;
    for(++row; row < data.length; ++row) {
      if(data[row].level === this_level) {
        return true;
      }

      if(data[row].level < this_level) {
        break;
      }
    }

    return false;
  },

  toggleOpenState: function(row) {
    this._data[row].open = !this._data[row].open;
    this._doRefresh();
  },

  getParentIndex: function(row) {
    var this_level = this._data[row].level;
    for(--row; row >= 0; --row) {
      if(this._data[row].level < this_level) {
        break;
      }
    }

    return row;
  },

  isContainer: function(row) {
    return this._data[row].children.length > 0;
  },

  isContainerEmpty: function(row) {
    return this._data[row].children.length === 0;
  },

  isContainerOpen: function(row) {
    return this._data[row].open;
  },

  isSeparator: function(row) {
    return false;
  },

  isSorted: function() {
    return true;
  },

  getImageSrc: function(row, column) {
    return null;
  },

  getRowProperties: function(row, props) {},

  
  getCellProperties: function(row, column, props) {
    if(column.id === 'col_coverage') {
      var aserv=Cc["@mozilla.org/atom-service;1"].
        getService(Ci.nsIAtomService);

      var percent = this._data[row].coveragePercent;

      if(percent >= COV_GREAT) {
        props.AppendElement(aserv.getAtom('greatCoverage'));
      } else if(percent >= COV_GOOD) {
        props.AppendElement(aserv.getAtom('goodCoverage'));
      } else if(percent >= COV_MEDIOCRE) {
        props.AppendElement(aserv.getAtom('mediocreCoverage'));
      } else if(percent) {
        props.AppendElement(aserv.getAtom('badCoverage'));
      }
    }
  },
  
  getColumnProperties: function(colid, col, props) {}
};

// == Commands

function get_layout() {
  if(document.getElementById('details-top-menuitem')
     .getAttribute('checked') === 'true')
  {
    return 'top';
  }
  
  if(document.getElementById('details-left-menuitem')
            .getAttribute('checked') === 'true')
  {
    return 'left';
  }
  
  if(document.getElementById('details-right-menuitem')
            .getAttribute('checked') === 'true')
  {
    return 'right';
  }
  
  if(document.getElementById('details-bottom-menuitem')
          .getAttribute('checked') === 'true')
  {
    return 'bottom';
  }

  return null;
}

function update_layout() {
  var layout = get_layout();
  dump('layout:' + layout + '\n');
  gMainBox.setAttribute('layout', layout);

  // XUL bug. Don't ask me why doing this works.
  window.setTimeout(update_layout2, 0);
}

function update_layout2() {
 gMainBox.removeChild(gMainBox.firstChild.nextSibling);
  var splitter = document.createElementNS(NS_XUL, 'splitter');
  // broken for our use. doesn't like reversed boxes, for one
  // splitter.setAttribute('collapse', 'both');
  gMainBox.insertBefore(splitter, gMainBox.lastChild);
}

function results_onload() {
  gDeck = document.getElementById('deck');
  gTree = document.getElementById('results-tree');
  gRcCmd = document.getElementById('hrc_toggle_recording');
  gFilterBox = document.getElementById('filter');
  gFilterLabel = document.getElementById('filter-label');
  gMainBox = document.getElementById('main-box');

  gTree.view = gTreeView = new HRCovTreeView();

  gFilterBox.value = get_saved_filter();
  on_filter_input();
  
  document.getElementById('sourcett')
    .addEventListener('popupshowing', on_sourcett_showing, false);
  gTree.addEventListener('select', on_select, false);
  gFilterBox.addEventListener('input', on_filter_input, false);
  gFilterBox.addEventListener('change', on_filter_change, false);

  if(!get_layout()) {
    dump('Setting default layout');
    
    document.getElementById('details-right-menuitem')
      .setAttribute('checked', 'true');
  }
  
  update_layout();
}

function results_onunload() {
  if(gFetchInProgress) {
    gFetchInProgress.cancel();
    gFetchInProgress = null;
  }

  gHrc.stop();
}

function hrc_toggle_recording() {
  if(gDeck.selectedIndex == PAGE_RESULTS ||
     gDeck.selectedIndex == PAGE_INITIAL )
  {
    trans_results_to_recording();
  }
  else if(gDeck.selectedIndex == PAGE_RECORDING)
  {
    trans_recording_to_processing();
  }
}

function hrc_about_cmd() {
  gPrompts.alert(window, 'HRCov 1.3',
                 'HRCov 1.3 — Line-based code coverage for Javascript\n' +
                 '\n' +
                 '© 2008, 2009 Daniel Colascione');
}


// == Emacs Variables
// Local Variables:
// outline-regexp: "// ==+\\s-+"
// End:
