var gArguments = null;
var gDB;
var gFlatTree;
var gFlatTreeView;
var gCallersTree;
var gCallersTreeView;
var gCalleesTree;
var gCalleesTreeView;

/**
  * The treeview for the flat profile
  */
function ProfileTreeView(info, refresh_func, prefix, metadata) {
  this._info = info;
  this._refreshFunc = refresh_func;
  this._prefixLen = prefix.length;
  this._isCounter = metadata.timeSourceIsCounter;
  this.rowCount = info.length;
}

ProfileTreeView.prototype = {
  constructor: ProfileTreeView,
  
  getCellText: function(row, column) {
    var column_id = column.id.substring(this._prefixLen);
    var value = this._info[row][column_id];
    
    if(column_id.indexOf('seconds_') !== -1 &&
       !this._isCounter)
    {
      value = this._formatNanoSeconds(value);
    }

    return value;
  },

  _formatNanoSeconds: function(nsec) {
    var divisor, units;

    if(nsec >= 1e9) {
      divisor = 1e9;
      units = 's';
    } else if(nsec >= 1e6) {
      divisor = 1e6;
      units = 'ms';
    } else if(nsec >= 1e3) {
      divisor = 1e3;
      units = 'µs';
    } else {
      divisor = 1;
      units = 'ns';
    }

    return (nsec / divisor).toPrecision(3) + units;
  },

  cycleHeader: function(column) {
    var tree = this._treebox.treeBody.parentNode;
    
    if(tree.getAttribute('sortResource') === column.id) {
      if(tree.getAttribute('sortDirection') === 'ascending') {
        tree.setAttribute('sortDirection', 'descending');
      } else {
        tree.setAttribute('sortDirection', 'ascending');
      }
    } else {
      tree.setAttribute('sortResource', column.id);
      tree.setAttribute('sortDirection', 'ascending');
    }

    this._refreshFunc();
  },

  setTree: function(treebox) {
    this._treebox = treebox;
  },

  isContainer: function(row) {
    return false;
  },

  isSeparator: function(row) {
    return false;
  },

  isSorted: function() {
    return false;
  },

  getLevel: function() {
    return 0;
  },

  getImageSrc: function(row, col) {
    return null;
  },

  getRowProperties: function(row, props) {},
  getCellProperties: function(row, col, props) {},
  getColumnProperties: function(colid, col, props) {},

  getIDByRow: function(row) {
    return this._info[row].funcid;
  },

  getRowByID: function(funcid) {
    for(var row = 0; row < this._info.length; ++row) {
      if(this._info[row].funcid === funcid) {
        return row;
      }
    }

    return null;
  }
};

function update_window_title() {
  document.title = 'HRProfiler - ' +
    (gArguments.is_tmp ? '*' : '') +
    gArguments.db_file.path;
}

function hrp_results_onload() {
  if(!window.arguments) {
    gPrompts.alert(window, 'HRProfiler', 'Must be started with arguments');
    window.close();
    return;
  }

  try {
    gArguments = window.arguments[0];
    gDB = new HRPCallTree(gArguments.db_file);
    update_window_title();
    gFlatTree = document.getElementById('flat-profile-tree');
    gCalleesTree = document.getElementById('callees-profile-tree');
    gCallersTree = document.getElementById('callers-profile-tree');
    assert(function() gFlatTree);
    assert(function() gCalleesTree);
    assert(function() gCallersTree);
    hrp_refresh_flat();
    gFlatTree.addEventListener('select', hrp_on_flat_selection, false);
    gCallersTree.addEventListener('dblclick', hrp_on_subview_dblclick, false);
    gCalleesTree.addEventListener('dblclick', hrp_on_subview_dblclick, false);
    gCallersTree.addEventListener('keypress', hrp_on_subview_keypress, false);
    gCalleesTree.addEventListener('keypress', hrp_on_subview_keypress, false);

  } catch(ex) {
    gPrompts.alert(window, 'HRProfiler', ex);
    window.close();
  }
}

function ignore_event(e) {
  e.preventDefault();
}

var FLAT_LENGTH = 'flat_'.length;

function hrp_refresh_flat() {
  var sort_resource, sort_direction;
  [sort_resource, sort_direction] = update_tree_sorting(gFlatTree);

  var flat_profile = gDB.getFlatProfile(sort_resource.substring(FLAT_LENGTH));

  if(sort_direction !== 'ascending') {
    flat_profile.reverse();
  }

  gFlatTree.view = gFlatTreeView = new ProfileTreeView(
    flat_profile, hrp_refresh_flat, 'flat_', gDB.getMetaData());

  hrp_refresh_callers();
  hrp_refresh_callees();
}

// Returns func_ids for the current selection in the flat
// profile view
function extract_row_info() {
  var selection = gFlatTree.view.selection;
  var num_ranges = selection.getRangeCount();
  var min = {};
  var max = {};
  var func_ids = [];

  for(var rangeno = 0; rangeno < num_ranges; ++rangeno) {
    selection.getRangeAt(rangeno, min, max);
    
    for(var rowno = min.value; rowno <= max.value; ++rowno) {
      var row = gFlatTreeView._info[rowno];
      func_ids.push(row.funcid);
    }
  }
  
  return func_ids;
}

function postprocess_subresult(results) {
  var num = results.length;
  for(var i = 0; i < num; ++i) {
    var row = results[i];
    var overall_info = gDB.getOverallFuncInfo(row.funcid);
    row.calls = row.calls + '/' + overall_info.calls;
  }
}

var CALLERS_LENGTH = 'callers_'.length;

function hrp_refresh_callers() {
  var func_ids = extract_row_info();
  
  if(func_ids.length) {
    gCallersTree.disabled = false;
    var sort_resource, sort_direction;
    [sort_resource, sort_direction] = update_tree_sorting(gCallersTree);

    var callers_profile = gDB.getCallersProfile(
      sort_resource.substring(CALLERS_LENGTH), func_ids);

    if(sort_direction !== 'ascending') {
      callers_profile.reverse();
    }

    postprocess_subresult(callers_profile);
  
    gCallersTree.view = gCallersTreeView = new ProfileTreeView(
      callers_profile, hrp_refresh_callers, 'callers_',
      gDB.getMetaData());
  } else {
    gCallersTree.disabled = true;
    gCallersTree.view = gCallersTreeView = new ProfileTreeView(
      [], hrp_refresh_callers, 'callers_',
      gDB.getMetaData());
  }
}

var CALLEES_LENGTH = 'callees_'.length;

function hrp_refresh_callees() {
  var func_ids = extract_row_info();

  if(func_ids.length) {
    gCalleesTree.disabled = false;
    var sort_resource, sort_direction;
    [sort_resource, sort_direction] = update_tree_sorting(gCalleesTree);

    var callees_profile = gDB.getCalleesProfile(
      sort_resource.substring(CALLEES_LENGTH), func_ids);

    if(sort_direction !== 'ascending') {
      callees_profile.reverse();
    }

    postprocess_subresult(callees_profile);
  
    gCalleesTree.view = gCalleesTreeView = new ProfileTreeView(
      callees_profile, hrp_refresh_callees, 'callees_',
      gDB.getMetaData());
  } else {
    gCalleesTree.disabled = true;
    gCalleesTree.view = gCalleesTreeView = new ProfileTreeView(
      [], hrp_refresh_callees, 'callees_',
      gDB.getMetaData());
  }
}

function hrp_on_flat_selection() {
  window.setCursor('wait');
  pump_events();
  
  try {
    hrp_refresh_callers();
    hrp_refresh_callees();
  } finally {
    window.setCursor('auto');
  }
}

function hrp_goto_function(funcid) {
  var flat_rowno = gFlatTreeView.getRowByID(funcid);
  gFlatTree.view.selection.select(flat_rowno);
  gFlatTreeView._treebox.ensureRowIsVisible(flat_rowno);
}

function hrp_on_subview_dblclick(e) {
  var tree = e.target;

  if(tree.parentNode === gCallersTree) {
    var view = gCallersTreeView;
  } else {
    var view = gCalleesTreeView;
  }

  var row = {};
  view._treebox.getCellAt(e.clientX, e.clientY, row, {}, {});
  row = row.value;
  if(row >= 0) {
    hrp_goto_function(view.getIDByRow(row));
  }  
}

function hrp_on_subview_keypress(e) {
  var tree = e.target;
  if(tree === gCallersTree) {
    var view = gCallersTreeView;
  } else {
    var view = gCalleesTreeView;
  }

  dump('e.keyCode: ' + e.keyCode + '\n');
  if(e.keyCode === KeyEvent.DOM_VK_ENTER ||
     e.keyCode === KeyEvent.DOM_VK_RETURN)
  {
    var row = tree.currentIndex;
    if(row >= 0) {
      hrp_goto_function(view.getIDByRow(row));
    }
  }
}

function hrp_confirm_close_unsaved() {
  return gPrompts.confirm(window, "HRProfiler",
                          "Profiling information has not been saved and will be lost." +
                          " Continue?");
}

function hrp_close_cmd() {
  if(gArguments.is_tmp && !hrp_confirm_close_unsaved()) {
    return;
  }

  window.close();
}

function hrp_results_handleclose(e) {
  if(gArguments.is_tmp && !hrp_confirm_close_unsaved())
  {
    e.preventDefault();
  }
}

function hrp_results_unonload() {
  if(gDB) {
    gDB.close();
  }

  if(gArguments.is_tmp) {
    gArguments.db_file.remove(false /*nonrecursive*/);
  }
}

/**
  * Copy the nsIFile input_name to the nsIFile output_name, creating
  * the latter if it does not exist and overwriting it if it does.
  * Show a progress dialog.
  *
  * If the user cancels, or if an exception is thrown, unlink
  * output_name if it was successfully opened.
  *
  * @return: true if copy succeeded, false if the user canceled
  */
function copy_file_with_progress(input_name, output_name) {
  var input = null;
  var output = null;
  var success = false;

  try {
    var input = open_input_stream(input_name, PR_RDONLY);
    var output = open_output_stream(output_name,
                                    PR_WRONLY | PR_CREATE_FILE | PR_TRUNCATE,
                                    0600);
    
    var ss = input.QueryInterface(Ci.nsISeekableStream);

    ss.seek(2, 0);
    var total_size = ss.tell();
    ss.seek(0, 0);

    var chunk_size = 4096;

    // XXX: Really should use a stream pump for this kind of thing, I guess
    
    var binary_input = Cc["@mozilla.org/binaryinputstream;1"]
      .createInstance(Ci.nsIBinaryInputStream);
    binary_input.setInputStream(input);

    var binary_output = Cc["@mozilla.org/binaryoutputstream;1"]
      .createInstance(Ci.nsIBinaryOutputStream);
    binary_output.setOutputStream(output);

    function iterator() {
      var bytes_left = total_size;
      while(bytes_left) {
        var this_chunk_size = Math.max(chunk_size, bytes_left);
        var bytes = binary_input.readByteArray(this_chunk_size);
        binary_output.writeByteArray(bytes, bytes.length);
        bytes_left -= this_chunk_size;
        yield true;
      }
    }
    
    var ret = {};

    window.openDialog('chrome://ffhrtimer/content/ffhrprofiler_progress.xul',
                      '_blank',
                      'modal,centerscreen',
                      {
                        iterator: iterator(),
                        get_progress_func: function() ss.tell(),
                        get_total_func: function() total_size,
                        message: "Saving…"
                      },
                      ret);

    if(!ret.canceled) {
      success = true;
    }

  } finally {
    if(input) {
      input.close();
    }

    if(output) {
      output.close();
    }

    if(!success) {
      try {
        output_name.unlink(false /*non-recursive*/);
      } catch(ex) {
        ;
      }
    }
  }

  return success;
}

/**
  * Open a save dialog box with the given title and ask the user to
  * enter a file. If the user cancels, return false.
  *
  * Otherwise, copy the current database file to the one the user
  * picked, and return its nsIFile. If an exception is thrown, the new
  * file will be deleted.
  *
  */
function hrp_doSave(title) {
  var nsIFilePicker = Ci.nsIFilePicker;
  var fp = Cc["@mozilla.org/filepicker;1"].createInstance(nsIFilePicker);
  fp.init(window, title, nsIFilePicker.modeSave);
  fp.appendFilter("HRProfiler Files", "*.hrprofiler");
  var rv = fp.show();
  var ok = false;

  if(rv === nsIFilePicker.returnOK || rv === nsIFilePicker.returnReplace) {
    if(gArguments.db_file.equals(fp.file)) {
      gPrompts.alert(window, 'HRProfiler', 'Cannot save to same file');
      return false;
    }
    
    if(copy_file_with_progress(gArguments.db_file, fp.file)) {
      return fp.file;
    }
  }

  return false;
}

function hrp_saveAs_cmd() {
  try {
    var saved_file = hrp_doSave('HRProfiler - Save As');

    if(saved_file !== false) {
      var newdb = new HRPCallTree(saved_file);
      gDB.close();

      if(gArguments.is_tmp) {
        try {
          gArguments.db_file.remove(false /*nonrecursive*/);
        } catch(ex) {
          ;
        }
      }
      
      gDB = newdb;
      gArguments.db_file = saved_file;
      gArguments.is_tmp = false;
      update_window_title();
    }
    
  } catch(ex) {
    gPrompts.alert(window, 'HRProfiler', ex);
  }
}

function hrp_saveCopyAs_cmd() {
  try {
    hrp_doSave('HRProfiler - Save Copy As');
  } catch(ex) {
    gPrompts.alert(window, 'HRProfiler', ex);
  }
}

function hrp_about_cmd() {
  gPrompts.alert(window, 'HRProfiler 1.3',
                 'HRProfiler 1.3 — High-resolution profiler for Javascript\n' +
                 '\n' +
                 '© 2008, 2009 Daniel Colascione');
}

window.addEventListener('close', hrp_results_handleclose, false);
window.addEventListener('unload', hrp_results_unonload, false);
