#include <prthread.h>
#include <prlock.h>
#include <prmem.h>
#include "HRProfiler.h"
#include "HRTimer.h"
#include "nsVoidArray.h"
#include "nsCRT.h"
#include "nsCOMPtr.h"
#include "nsXPCOMCID.h"
#include "nsComponentManagerUtils.h"
#include "nsServiceManagerUtils.h"
#include "nsIServiceManager.h"
#include "nsIClassInfoImpl.h"
#include "common.h"
#include "jsinterp.h"
#include "jsfun.h"
#include "jsdbgapi.h"
#include "nsIJSRuntimeService.h"
#include "nsDataHashtable.h"
#include "nsBaseHashtable.h"
#include "nsIXPConnect.h"
#include "nsCycleCollectionParticipant.h"
#include "nsThreadUtils.h"
#include "nsIThread.h"

#define nsAString_h___ // work around bad header
#include "jsdIDebuggerService.h"


#define RECORD_CALLEE_IN_LOGS 0
#define DEBUG_GN 0

#define XPC_RUNTIME_CONTRACTID "@mozilla.org/js/xpc/RuntimeService;1"

static nsCString JSString_to_nsCString(JSString* s);
static nsCString error_message(JSContext* context);

// Attempt bytes bytes into buf. If we fail, return immediately with
// the failure code. On success, all bytes have been read; return
// NS_OK.
static nsresult read_all(nsIInputStream* input,
                         void* buf, size_t bytes, bool* at_eof);

/**
 * List of JS object class names to never inspect; we'll never find
 * anything interesting in them, and they're pretty expensive to
 * enumerate.
 */
static const char* blacklisted_object_classes[] = {
  "nsXPCComponents_Results",
  "nsXPCComponents_Interfaces",
  "nsXPCComponents_Classes"
};



#if DEBUG_GN
static int debug_gn_lvl = 0;

#define SPACES "                                                       "
#define DEBUG_GNP(format, ...) \
  fprintf(stderr, "%.*s" format "\n", debug_gn_lvl, SPACES , ## __VA_ARGS__)
#else
#define DEBUG_GNP(format, ...)
#endif

struct HRFuncInfo {
  nsCString name;
  nsCString filename;
  uintN     lineno;
  uintN     refcount;
};

static inline void release_funcinfo(HRFuncInfo* info) {
  PR_ASSERT(info->refcount >= 1);
  if(! --info->refcount) {
    delete info;
  }
}


#define HR_SEARCH_CHROME_WINDOW   (1<<0)
#define HR_SEARCH_WINDOW          (1<<1)
#define HR_SEARCH_OTHER           (1<<2)
#define HR_SEARCH_ONLY_PROTOTYPES (1<<3)
#define HR_PREFER_GIVEN_NAMES     (1<<4)
#define HR_MAIN_THREAD_ONLY       (1<<5)
#define HR_USE_FILELINE_CACHE     (1<<6)

#define HRENT_TYPE_CALL   0x1231
#define HRENT_TYPE_RETURN 0x1232
#define HRENT_TYPE_INFO   0x1233

#ifdef _MSC_VER
#pragma pack(push, 1)
#endif // _MSC_VER

// Header of every record written to the output file. Size is an
// HRENT_TYPE_* constant. (Chosen so they serve as sort-of magic
// numbers too.) Size is the size of the entry, in bytes, including
// the header.
//
// All strings are NULL-terminated UTF8. All multibyte values are
// little-endian. For some loads, the amount of disk IO to record all
// the calls is actually dominant, so try to make these structures as
// small as possible.
//

struct HREnt {
  PRUint16 type;
  PRUint32 size;
}
#ifdef __GNUC__
__attribute__((packed))
#endif
;

struct HRCallEnt  {
  HREnt    header;
  PRUint32 thread; // opaque thread disambiguator
#if RECORD_CALLEE_IN_LOGS
  PRUint32 callee;
#endif
  PRUint64 nanoseconds;
}
#ifdef __GNUC__
__attribute__((packed))
#endif
;


struct HRReturnEnt {
  HREnt    header;
  PRUint32 thread; // opaque thread disambiguator
#if RECORD_CALLEE_IN_LOGS
  PRUint32 callee;
#endif

  // profiler started processing here
  PRUint64 nanoseconds_start;

  // and ended processing here
  PRUint64 nanoseconds_end;
  PRUint32 lineno;

  // Name of block and filename as two zero-terminated UTF8 strings
  char     name_and_filename[1];
}
#ifdef __GNUC__
__attribute__((packed))
#endif
;

struct HRInfoEnt {
  HREnt    header;
  PRUint32 flags; // HR_SEARCH_*


  // zero-terminated name of clock source
  char     clockname[64];

  // search depth
  PRUint16 maxDepth;
  PRUint16 timeSourceIsCounter;
}
#ifdef __GNUC__
__attribute__((packed))
#endif
;

#ifdef _MSC_VER
#pragma pack(pop)
#endif // _MSC_VER

static HREnt* copy_ent(HREnt* ent) {
  HREnt* ret = (HREnt*)PR_Malloc(ent->size);

  if(likely(ret != NULL)) {
    memcpy(ret, ent, ent->size);
  }

  return ret;
}

struct HRTLI {
  public:
  bool          inhibited;

  HRTLI()
    : inhibited(false),
      buf(NULL),
      buf_size(0)
  {}

  ~HRTLI() {
    PR_Free(buf);
  }

  /**
   * Make sure this thread's temporary buffer is at least req_size
   * bytes and return the buffer.
   */
  void* alloc_tmpbuf(size_t req_size) {
    if(unlikely(req_size == 0)) {
      req_size = 1;
    }

    if(likely(req_size <= buf_size)) {
      return buf;
    }

    void* tmpbuf = PR_Realloc(buf, req_size);
    if(likely(tmpbuf != NULL)) {
      buf = tmpbuf;
      buf_size = req_size;
    } else {
      NS_ERROR("could not allocate tmpbuf!");
    }

    return tmpbuf;
  }

  private:
  void*         buf;
  size_t        buf_size;
};

class HRProfiler : public IHRProfiler
{
  public:
  NS_DECL_CYCLE_COLLECTING_ISUPPORTS
  NS_DECL_IHRPROFILER
  NS_DECL_CYCLE_COLLECTION_CLASS(HRProfiler)

  HRProfiler();

private:
  ~HRProfiler();

  /* Data */
  static HRProfiler* sInst;
  PRUint32 mFlags;
  PRUint16 mMaxDepth;
  PRBool mWasOn;
  nsCOMPtr<IHRTimer> mTimer;

  friend NS_IMETHODIMP HRProfilerSingletonConstructor(nsISupports *aOuter,
                                                      const nsIID& aIID,
                                                      void **aResult);



  // This is such a hack... we're enabled via script. If debugging is
  // enabled, spidermonkey will have called the existing callhook on
  // the way down. When we change the call hook, our hook gets called
  // for the return-side, but with the wrong closure.
  //
  // But not to be fooled, we notice the mismatch between our closure
  // and HRProfiler::sInst. We store the mistaken closure in
  // mUglyClosure. When we stop profiling later, EndProfiling will
  // turn JSD back on, if it was disabled. When that happens, JSD's
  // CallHook will be called with our closure and crash. To avoid
  // that, we always return the JSD closure.
  void* mUglyClosure;

  // maps function objects to information about these functions
  // pointers are owned by mFunctionCache. Protected by mLock.
  nsDataHashtable<nsVoidPtrHashKey, HRFuncInfo*> mFunctionCache;
  nsDataHashtable<nsCharPtrHashKey, HRFuncInfo*> mFileLineCache;

  nsCOMPtr<nsIXPConnect> mXPConnect;
  PRThread* mMainThread;
  PRUintn mThreadPrivateIndex;

  // protected by mLock
  nsCOMPtr<nsIOutputStream> mOutput;
  PRLock* mLock;

  /* Private functions */
  void OnFunctionCall(JSContext* cx, JSStackFrame* fp);
  void OnFunctionReturn(JSContext* cx, JSStackFrame* fp);
  void OnObjectCreated(JSContext* cx, JSObject* obj);

  static void* CallHook(JSContext *cx, JSStackFrame *fp, JSBool before,
                        JSBool *ok, void *closure);

  static void ObjectHook(JSContext* cx, JSObject* obj, JSBool isNew,
                         void* closure);

  nsCString GuessName(JSContext* cx, JSStackFrame* fp);

  bool SearchJSValIteration(nsCString& name,
                            JSContext* context,
                            JSObject* obj,
                            jsval search,
                            ptr_set& visited_objects,
                            unsigned int depth);

  /**
   * Search all the properties of scope that actually belong to scope,
   * as determined by JS_AlreadyHasOwnUCProperty.
   */
  bool SearchForJSVal(nsCString& name,
                      JSContext* context,
                      JSObject* scope,
                      jsval search,
                      ptr_set& visited_objects,
                      unsigned int depth);

  /**
   * Search the built-in list of global names on scope.
   */
  bool SearchGlobalNames(nsCString& name,
                         JSContext* context,
                         JSObject* scope,
                         jsval search,
                         ptr_set& visited_objects);

  // Thread-safe functions
  void WriteEnt(HREnt* ent);
  const HRFuncInfo* GetFuncInfo(JSContext* cx, JSStackFrame* fp, bool* cached);
  inline HRTLI* GetTLI();
};

HRProfiler* HRProfiler::sInst = NULL;

PR_STATIC_CALLBACK(void) tl_dtor(void* priv) {
  delete (HRTLI*)priv;
}

NS_IMETHODIMP HRProfilerSingletonConstructor(nsISupports *aOuter,
                                             const nsIID& aIID,
                                             void **aResult)
{
  if(unlikely(aOuter != NULL)) {
    return ((nsresult) 0x80040110L);;
  }

  if(likely(HRProfiler::sInst != NULL)) {
    return HRProfiler::sInst->QueryInterface(aIID, aResult);
  }

  HRProfiler::sInst = new HRProfiler();
  if(!HRProfiler::sInst) {
    return NS_ERROR_OUT_OF_MEMORY;
  }

  nsCOMPtr<IHRProfiler> guard = HRProfiler::sInst;

  HRProfiler* hrp = HRProfiler::sInst;

  hrp->mFlags =
    HR_SEARCH_WINDOW | HR_SEARCH_CHROME_WINDOW | HR_SEARCH_OTHER |
    HR_PREFER_GIVEN_NAMES | HR_USE_FILELINE_CACHE |
    HR_SEARCH_ONLY_PROTOTYPES | HR_MAIN_THREAD_ONLY;

  NS_ENSURE_TRUE(hrp->mFunctionCache.Init(1), NS_ERROR_UNEXPECTED);
  NS_ENSURE_TRUE(hrp->mFileLineCache.Init(1), NS_ERROR_UNEXPECTED);

  nsresult rv = NS_ERROR_UNEXPECTED;
  hrp->mXPConnect = do_GetService("@mozilla.org/js/xpc/XPConnect;1", &rv);
  NS_ENSURE_TRUE(hrp->mXPConnect, rv);

  nsCOMPtr<nsIThread> com_main_thread;
  NS_GetMainThread(getter_AddRefs(com_main_thread));
  NS_ENSURE_TRUE(com_main_thread, NS_ERROR_UNEXPECTED);

  rv = com_main_thread->GetPRThread(&hrp->mMainThread);
  NS_ENSURE_SUCCESS(rv, rv);

  hrp->mMaxDepth = 1;
  hrp->mWasOn = false;

  PRStatus s = PR_NewThreadPrivateIndex(&hrp->mThreadPrivateIndex, tl_dtor);
  NS_ENSURE_TRUE(s != PR_FAILURE, NS_ERROR_UNEXPECTED);

  hrp->mLock = PR_NewLock();
  NS_ENSURE_TRUE(hrp->mLock, NS_ERROR_UNEXPECTED);

  NS_ADDREF(hrp);

#if DEBUG
  fprintf(stderr, "HRProfiler initialized\n");
#endif

  return HRProfiler::sInst->QueryInterface(aIID, aResult);
}

NS_IMPL_CYCLE_COLLECTION_2(HRProfiler, mOutput, mTimer)
NS_INTERFACE_MAP_BEGIN_CYCLE_COLLECTION(HRProfiler)
NS_INTERFACE_MAP_ENTRY(IHRProfiler)
NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, IHRProfiler)
NS_IMPL_QUERY_CLASSINFO(HRProfiler)
NS_INTERFACE_MAP_END
NS_IMPL_CI_INTERFACE_GETTER1(HRProfiler, IHRProfiler)

NS_IMPL_CYCLE_COLLECTING_ADDREF(HRProfiler)
NS_IMPL_CYCLE_COLLECTING_RELEASE(HRProfiler)

HRProfiler::HRProfiler()
{ /* Real work done in the singleton constructor above */ }

HRProfiler::~HRProfiler()
{
  PR_DestroyLock(mLock);
  fprintf(stderr, "HRProfiler destroyed\n");
}

HRTLI* HRProfiler::GetTLI() {
  HRTLI* tli = (HRTLI*)PR_GetThreadPrivate(mThreadPrivateIndex);
  if(!tli) {
    tli = new HRTLI;
    PR_SetThreadPrivate(mThreadPrivateIndex, tli);
  }

  return tli;
}

/* readonly attribute PRBool profiling; */
NS_IMETHODIMP HRProfiler::GetProfiling(PRBool *aProfiling)
{
  *aProfiling = !!mOutput;
  return NS_OK;
}

static PLDHashOperator free_enumerator(const void* key,
                                       HRFuncInfo* value,
                                       void* ignored)
{
  release_funcinfo(value);
  return PL_DHASH_NEXT;
}

static PLDHashOperator free_enumerator_str(const char* key,
                                           HRFuncInfo* value,
                                           void* ignored)
{
  release_funcinfo(value);
  return PL_DHASH_NEXT;
}

/* attribute boolean searchChromeWindow; */
NS_IMETHODIMP HRProfiler::GetSearchChromeWindow(PRBool *aSearchChromeWindow)
{
  *aSearchChromeWindow = mFlags & HR_SEARCH_CHROME_WINDOW;
  return NS_OK;
}
NS_IMETHODIMP HRProfiler::SetSearchChromeWindow(PRBool aSearchChromeWindow)
{
  if(aSearchChromeWindow) {
    mFlags |= HR_SEARCH_CHROME_WINDOW;
  } else {
    mFlags &= ~HR_SEARCH_CHROME_WINDOW;
  }

  return NS_OK;
}

/* attribute boolean searchWindow; */
NS_IMETHODIMP HRProfiler::GetSearchWindow(PRBool *aSearchWindow)
{
  *aSearchWindow = mFlags & HR_SEARCH_WINDOW;
  return NS_OK;
}
NS_IMETHODIMP HRProfiler::SetSearchWindow(PRBool aSearchWindow)
{
  if(aSearchWindow) {
    mFlags |= HR_SEARCH_WINDOW;
  } else {
    mFlags &= ~HR_SEARCH_WINDOW;
  }

  return NS_OK;
}

/* attribute boolean searchOther; */
NS_IMETHODIMP HRProfiler::GetSearchOther(PRBool *aSearchOther)
{
  *aSearchOther = mFlags & HR_SEARCH_OTHER;
  return NS_OK;
}
NS_IMETHODIMP HRProfiler::SetSearchOther(PRBool aSearchOther)
{
  if(aSearchOther) {
    mFlags |= HR_SEARCH_OTHER;
  } else {
    mFlags &= ~HR_SEARCH_OTHER;
  }

  return NS_OK;
}

/* attribute boolean searchOnlyPrototypes; */
NS_IMETHODIMP HRProfiler::GetSearchOnlyPrototypes(PRBool *aSearchOnlyPrototypes)
{
  *aSearchOnlyPrototypes = mFlags & HR_SEARCH_ONLY_PROTOTYPES;
  return NS_OK;
}
NS_IMETHODIMP HRProfiler::SetSearchOnlyPrototypes(PRBool aSearchOnlyPrototypes)
{
  if(aSearchOnlyPrototypes) {
    mFlags |= HR_SEARCH_ONLY_PROTOTYPES;
  } else {
    mFlags &= ~HR_SEARCH_ONLY_PROTOTYPES;
  }

  return NS_OK;
}

/* attribute boolean preferGivenNames; */
NS_IMETHODIMP HRProfiler::GetPreferGivenNames(PRBool *aPreferGivenNames)
{
  *aPreferGivenNames = mFlags & HR_PREFER_GIVEN_NAMES;
  return NS_OK;
}
NS_IMETHODIMP HRProfiler::SetPreferGivenNames(PRBool aPreferGivenNames)
{
  if(aPreferGivenNames) {
    mFlags |= HR_PREFER_GIVEN_NAMES;
  } else {
    mFlags &= ~HR_PREFER_GIVEN_NAMES;
  }

  return NS_OK;
}

/* attribute boolean mainThreadOnly; */
NS_IMETHODIMP HRProfiler::GetMainThreadOnly(PRBool *aMainThreadOnly)
{
  *aMainThreadOnly = mFlags & HR_MAIN_THREAD_ONLY;
  return NS_OK;
}
NS_IMETHODIMP HRProfiler::SetMainThreadOnly(PRBool aMainThreadOnly)
{
  if(aMainThreadOnly) {
    mFlags |= HR_MAIN_THREAD_ONLY;
  } else {
    mFlags &= ~HR_MAIN_THREAD_ONLY;
  }

  return NS_OK;
}

/* attribute boolean useFileLineCache; */
NS_IMETHODIMP HRProfiler::GetUseFileLineCache(PRBool *aUseFileLineCache)
{
  *aUseFileLineCache = mFlags & HR_USE_FILELINE_CACHE;
  return NS_OK;
}
NS_IMETHODIMP HRProfiler::SetUseFileLineCache(PRBool aUseFileLineCache)
{
  if(aUseFileLineCache) {
    mFlags |= HR_USE_FILELINE_CACHE;
  } else {
    mFlags &= ~HR_USE_FILELINE_CACHE;
  }

  return NS_OK;
}


/* attribute unsigned short maxDepth; */
NS_IMETHODIMP HRProfiler::GetMaxDepth(PRUint16 *aMaxDepth)
{
  *aMaxDepth = mMaxDepth;
  return NS_OK;
}
NS_IMETHODIMP HRProfiler::SetMaxDepth(PRUint16 aMaxDepth)
{
  mMaxDepth = aMaxDepth;
  return NS_OK;
}

/* void startProfiling (); */
NS_IMETHODIMP HRProfiler::StartProfiling(nsIOutputStream* output,
                                         IHRTimer* timer)
{
  nsresult rv;

  if(!output || !timer) {
    return NS_ERROR_NULL_POINTER;
  }

  if(mOutput) {
    return NS_ERROR_ALREADY_INITIALIZED;
  }

  nsCOMPtr<nsIJSRuntimeService> runtimeService = do_GetService(
    XPC_RUNTIME_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  JSRuntime* runtime;
  rv = runtimeService->GetRuntime(&runtime);
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<jsdIDebuggerService> debugger_service =
    do_GetService("@mozilla.org/js/jsd/debugger-service;1");
  if(debugger_service) {
    rv = debugger_service->GetIsOn(&mWasOn);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = debugger_service->Off();
    NS_ENSURE_SUCCESS(rv, rv);
#ifdef DEBUG
    fprintf(stderr, "debugger temporarily stopped\n");
#endif
  }

  JSDebugHooks* hooks = JS_GetGlobalDebugHooks(runtime);
  if(hooks->callHook || hooks->objectHook) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  nsCString clock_name;
  rv = timer->GetName(getter_Copies(clock_name));
  NS_ENSURE_SUCCESS(rv, rv);

  mFunctionCache.EnumerateRead(free_enumerator, NULL);
  mFunctionCache.Clear();

  mFileLineCache.EnumerateRead(free_enumerator_str, NULL);
  mFileLineCache.Clear();

  mTimer = timer;
  mOutput = output;

  struct HRInfoEnt ent;
  memset(&ent, 0, sizeof(ent));
  ent.header.type = HRENT_TYPE_INFO;
  ent.header.size = sizeof(ent);
  ent.flags = mFlags;

  PRBool is_counter;
  timer->GetIsCounter(&is_counter);

  ent.timeSourceIsCounter = !!is_counter;
  ent.maxDepth = mMaxDepth;

  strncpy(ent.clockname, clock_name.get(), sizeof(ent.clockname) - 1);
  WriteEnt((HREnt*)&ent);

  mUglyClosure = this;
  JS_SetCallHook(runtime, HRProfiler::CallHook, HRProfiler::sInst);
  JS_SetObjectHook(runtime, HRProfiler::ObjectHook, HRProfiler::sInst);

  return NS_OK;
}

/* nsIVariant endProfiling (); */
NS_IMETHODIMP HRProfiler::EndProfiling()
{
  nsresult rv;
  nsCOMPtr<nsIJSRuntimeService> runtimeService = do_GetService(
    XPC_RUNTIME_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  JSRuntime* runtime;
  rv = runtimeService->GetRuntime(&runtime);
  NS_ENSURE_SUCCESS(rv, rv);

  JSDebugHooks* hooks = JS_GetGlobalDebugHooks(runtime);
  if(hooks->callHook != HRProfiler::CallHook) {
    return NS_ERROR_UNEXPECTED;
  }

  PR_ASSERT(hooks->objectHook == HRProfiler::ObjectHook);

  JS_SetCallHook(runtime, NULL, NULL);
  JS_SetObjectHook(runtime, NULL, NULL);

  mOutput = NULL;
  mTimer = NULL;

  if(mWasOn) {
#ifdef DEBUG
    fprintf(stderr, "installing JSD hooks\n");
#endif

    nsCOMPtr<jsdIDebuggerService> debugger_service =
      do_GetService("@mozilla.org/js/jsd/debugger-service;1");

    if(debugger_service) {
      PRBool isOn;
      rv = debugger_service->GetIsOn(&isOn);
      NS_ABORT_IF_FALSE(NS_SUCCEEDED(rv), "could not get debugging status");

      if(isOn) {
        NS_WARNING("DEBUGGER SHOULD NOT HAVE BEEN RE-ENABLED");
      } else {
        rv = debugger_service->OnForRuntime(runtime);
        if(!NS_SUCCEEDED(rv)) {
          NS_WARNING("Could not re-enable debugger service");
        }
      }
    }
  }

  return NS_OK;
}

void HRProfiler::OnFunctionCall(JSContext* cx, JSStackFrame* fp) {
  // call context is incomplete here, so all we can do is add a call
  // entry and let the user sort it out
  PRThread* current_thread = PR_GetCurrentThread();
  if(mFlags & HR_MAIN_THREAD_ONLY && current_thread != mMainThread) {
    return;
  }

  HRTLI* tli = GetTLI();
  if(tli->inhibited) {
    return;
  }

  tli->inhibited = true;

  struct HRCallEnt ent;
  memset(&ent, 0, sizeof(ent));
  ent.header.type = HRENT_TYPE_CALL;
  ent.header.size = sizeof(ent);
  ent.thread = NS_PTR_TO_UINT32(current_thread);

#if RECORD_CALLEE_IN_LOGS
  ent.callee = NS_PTR_TO_UINT32(JS_GetFrameCalleeObject(cx, fp));
#endif

  mTimer->GetTime(&ent.nanoseconds);
  WriteEnt((HREnt*)&ent);

  tli->inhibited = false;
}

void HRProfiler::OnFunctionReturn(JSContext* cx, JSStackFrame* fp) {
  PRThread* current_thread = PR_GetCurrentThread();
  if(mFlags & HR_MAIN_THREAD_ONLY && current_thread != mMainThread) {
    return;
  }

  PRUint64 nanoseconds_start = 0;
  mTimer->GetTime(&nanoseconds_start);

  HRTLI* tli = GetTLI();
  if(tli->inhibited) {
    return;
  }

  tli->inhibited = true;

  bool was_cached;
  const HRFuncInfo* info = GetFuncInfo(cx, fp, &was_cached);

  if(unlikely(!info)) {
    NS_ERROR("could not allocate func info");
    return;
  }

#if DEBUG_GN
  if(!was_cached) {
    fprintf(stderr, "cache miss:\"%s\"@%s:%d\n",
            info->name.get(),
            info->filename.get(),
            info->lineno);
  }
#endif

  const char* name_start;
  PRUint32 name_len = info->name.BeginReading(&name_start, NULL);

  const char* filename_start;
  PRUint32 filename_len = info->filename.BeginReading(&filename_start, NULL);

  // + 2, one for each of the NULL terminators on name and filename.
  // Since the flexible array is declared with size 1 and we zero out
  // the whole structure on allocation, the structure is actually
  // double-null-terminated.
  size_t ent_size = sizeof(HRReturnEnt) + name_len + filename_len + 2;
  HRReturnEnt* ent = (HRReturnEnt*)tli->alloc_tmpbuf(ent_size);
  if(unlikely(!ent)) {
    NS_ERROR("could not allocate tmpbuf");
    tli->inhibited = false;
    return;
  }

  memset(ent, 0, ent_size);
  ent->header.type = HRENT_TYPE_RETURN;
  ent->header.size = ent_size;
  ent->thread = NS_PTR_TO_UINT32(current_thread);
#if RECORD_CALLEE_IN_LOGS
  ent->callee = NS_PTR_TO_UINT32(JS_GetFrameCalleeObject(cx, fp));
#endif
  ent->lineno = info->lineno;

  ent->nanoseconds_start = nanoseconds_start;

  memcpy(ent->name_and_filename, name_start, name_len);
  ent->name_and_filename[name_len] = 0;
  memcpy(ent->name_and_filename + name_len + 1,
         filename_start, filename_len);
  // Already null terminated by the memset

  mTimer->GetTime(&ent->nanoseconds_end);
  WriteEnt((HREnt*)ent);

  tli->inhibited = false;
}

void HRProfiler::OnObjectCreated(JSContext* cx, JSObject* obj)
{
  PRThread* current_thread = PR_GetCurrentThread();
  const bool main_thread_only = mFlags & HR_MAIN_THREAD_ONLY;
  if(main_thread_only && current_thread != mMainThread) {
    return;
  }

  HRFuncInfo* info = NULL;
  PR_ASSERT(obj);

  if(!main_thread_only) {
    PR_Lock(mLock);
  }

  if(obj && mFunctionCache.Get((void*)obj, &info)) {
    release_funcinfo(info);
    mFunctionCache.Remove((void*)obj);
  }

  if(!main_thread_only) {
    PR_Unlock(mLock);
  }
}

static nsCString JSString_to_nsCString(JSString* s) {
  return NS_ConvertUTF16toUTF8((PRUnichar*)JS_GetStringChars(s),
                               JS_GetStringLength(s));
}

nsCString hr_object_class_name(JSContext* context, JSObject* obj) {
  JSClass* cls = JS_GET_CLASS(context, obj);
  nsCString name;
  if(cls && cls->name) {
    name = cls->name;
  } else {
    name = NS_LITERAL_CSTRING("Unknown");
  }

  return name;
}

void* HRProfiler::CallHook(
  JSContext *cx,
  JSStackFrame *fp,
  JSBool before,
  JSBool *ok,
  void *closure)
{
  if(closure != HRProfiler::sInst) {
    HRProfiler::sInst->mUglyClosure = closure;
  }

  if(before) {
    HRProfiler::sInst->OnFunctionCall(cx, fp);
  } else {
    HRProfiler::sInst->OnFunctionReturn(cx, fp);
  }

  return HRProfiler::sInst->mUglyClosure;
}

void HRProfiler::ObjectHook(JSContext* cx, JSObject* obj, JSBool isNew,
                            void* closure)
{
  if(closure != HRProfiler::sInst) {
    NS_WARNING("BAD CALL");
  }

  if(isNew) {
    HRProfiler::sInst->OnObjectCreated(cx, obj);
  }
}

bool HRProfiler::SearchForJSVal(nsCString&   name,
                                JSContext*   context,
                                JSObject*    scope,
                                jsval        search,
                                ptr_set&     visited_objects,
                                unsigned int depth)
{
  bool found = false;
  const bool search_only_prototypes = mFlags & HR_SEARCH_ONLY_PROTOTYPES;
  jsval propval = JSVAL_VOID;

  PR_ASSERT(context);
  PR_ASSERT(scope);
  PR_ASSERT(search);

  visited_objects.PutEntry((void*)scope);

  JSIdArray* ids = JS_Enumerate(context, scope);
  if(ids) {
    // Root each of the properties in case we end up calling into JS
    // (and the GC) below
    for(jsint i = 0; i < ids->length; ++i) {
      JS_AddNamedRoot(context, &ids->vector[i], "SearchForJSVal");
    }
    JS_AddNamedRoot(context, &propval, "SearchForJSVal_propval");

    // From here on, control needs to flow through `out'

    for(jsint i = 0; i < ids->length; ++i) {
      // Don't need to separately gc-protect this because it's
      // initialized from a property, and that is already protected
      jsval propname;
      if(!JS_IdToValue(context, ids->vector[i], &propname)) {
        DEBUG_GNP("could not convert id to value");
        continue;
      }

#if DEBUG_GN
      DEBUG_GNP("property \"%s\"",
                JS_GetStringBytes(
                  JS_ValueToString(context, propname)));
#endif

      // We don't care about functions that might be hiding in numeric
      // properties
      if(!JSVAL_IS_STRING(propname)) {
        continue;
      }

      JSString* spropname = JSVAL_TO_STRING(propname);

      // We only care about properties that our scope object actually
      // owns
      JSBool has_own_property;
      if(!JS_AlreadyHasOwnUCProperty(context, scope,
                                     JS_GetStringChars(spropname),
                                     JS_GetStringLength(spropname),
                                     &has_own_property))
      {
        DEBUG_GNP("JS_AlreadyHasOwnUCProperty failed, abandoning: %s",
                  error_message(context).get());
        JS_ClearPendingException(context);
        goto out;
      }

      if(!has_own_property)
      {
        DEBUG_GNP("property \"%s\" but is not own",
                  JS_GetStringBytes(spropname));
        continue;
      }


      if(!JS_LookupUCProperty(context, scope,
                              JS_GetStringChars(spropname),
                              JS_GetStringLength(spropname),
                              &propval))
      {
        DEBUG_GNP("JS_LookupUCProperty failed, abandoning: %s",
                  error_message(context).get());
        JS_ClearPendingException(context);
        goto out;
      }

      if(propval == search) {
        DEBUG_GNP("**FOUND**: %s",
                  JS_GetStringBytes(spropname));
        name = JSString_to_nsCString(spropname);
        found = true;
        goto out;
      }

      if(depth >= mMaxDepth) {
        continue;
      }

      /* After this point, if we've hit a suitable property, we
       * recursively look into it */

      if(search_only_prototypes) {
        if(VALUE_IS_FUNCTION(context, propval)) {
          jsval prototype;
          if(!JS_LookupProperty(context, JSVAL_TO_OBJECT(propval),
                                "prototype", &prototype))
          {
            DEBUG_GNP("JS_LookupProperty(\"prototype\") failed: %s",
                      error_message(context).get());
            JS_ClearPendingException(context);
            continue;
          }

          if(JSVAL_IS_PRIMITIVE(prototype)) {
            DEBUG_GNP("prop \"%s\" is func, but has primitive prototype",
                      JS_GetStringBytes(spropname));
            continue;
          }

          JSObject* protoobj = JSVAL_TO_OBJECT(prototype);

          // Don't follow infinite object loops
          if(visited_objects.GetEntry((void*)protoobj)) {
            DEBUG_GNP("already visited proto of \"%s\"",
                      JS_GetStringBytes(spropname));
            continue;
          }

          DEBUG_GNP("looking at prototype of \"%s\"",
                    JS_GetStringBytes(spropname));
#if DEBUG_GN
          debug_gn_lvl += 4;
#endif
          found = SearchForJSVal(
            name, context, JSVAL_TO_OBJECT(prototype),
            search, visited_objects, depth + 1);
#if DEBUG_GN
          debug_gn_lvl -= 4;
#endif
        } else {
          DEBUG_GNP("property \"%s\" is not a function",
                    JS_GetStringBytes(spropname));
        }
      } else if(!JSVAL_IS_PRIMITIVE(propval)) {
        JSObject* prop_object = JSVAL_TO_OBJECT(propval);
        PR_ASSERT(prop_object);

        nsCString prop_cls = hr_object_class_name(context, prop_object);
        bool blacklisted = false;
        for(unsigned bl = 0;
            bl < NS_ARRAY_LENGTH(blacklisted_object_classes);
            ++bl)
        {
          if(prop_cls == blacklisted_object_classes[bl]) {
            blacklisted = true;
            break;
          }
        }

        if(blacklisted) {
          DEBUG_GNP("property \"%s\" (cls:%s) blacklisted",
                    JS_GetStringBytes(spropname),
                    hr_object_class_name(context, prop_object).get());
          continue;
        }

        // Don't follow infinite object loops
        if(visited_objects.GetEntry((void*)prop_object)) {
          continue;
        }
#if DEBUG_GN
        DEBUG_GNP("property \"%s\" (cls:%s)",
                  JS_GetStringBytes(spropname),
                  hr_object_class_name(context, prop_object).get());

        // search this object's properties
        debug_gn_lvl += 4;
#endif
        found = SearchForJSVal(name, context, prop_object, search,
                                 visited_objects, depth + 1);
#if DEBUG_GN
        debug_gn_lvl -= 4;
#endif
      }

      if(found) {
        nsCString prefix = JSString_to_nsCString(spropname);
        prefix += '.';

        name.Insert(prefix, 0);
        break;
      }
    }
  } else {
    DEBUG_GNP("JS_Enumerate failed: %s",
              error_message(context).get());
    JS_ClearPendingException(context);
  }

  out:
  JS_RemoveRoot(context, &propval);

  if(ids) {
    for(jsint i = 0; i < ids->length; ++i) {
      JS_RemoveRoot(context, &ids->vector[i]);
    }
    JS_DestroyIdArray(context, ids);
  }

  return found;
}

bool HRProfiler::SearchGlobalNames(nsCString&   name,
                                   JSContext*   context,
                                   JSObject*    scope,
                                   jsval        search,
                                   ptr_set&     visited_objects)
{
  const bool search_only_prototypes = mFlags & HR_SEARCH_ONLY_PROTOTYPES;
  enum { FOUND_NONE = 0, FOUND_PROTO, FOUND_NORMAL } found = FOUND_NONE;
  const HRGlobalNameEnt* gent = HRGlobalNames;

  /* Try searching through the builtin names */
  for(; gent->name; ++gent) {
#if DEBUG_GN
    nsCString dpropname = NS_ConvertUTF16toUTF8(
      (PRUnichar*)gent->name, gent->name_length);
#endif

    DEBUG_GNP("Searching built-in object \"%s\"",
              dpropname.get());

    jsval propval;
    if(!JS_LookupUCProperty(context, scope,
                            gent->name, gent->name_length,
                            &propval))
    {
      DEBUG_GNP("JS_GetUCProperty(\"%s\") failed, abandoning: %s",
                dpropname.get(),
                error_message(context).get());
      JS_ClearPendingException(context);
      return false;
    }

    if(propval == search) {
      DEBUG_GNP("**FOUND**: %s", dpropname.get());
      name = NS_ConvertUTF16toUTF8(
        (PRUnichar*)gent->name, gent->name_length);
      return true;
    }

    if(JSVAL_IS_PRIMITIVE(propval)) {
      DEBUG_GNP("built-in prop \"%s\" is primitive: %s",
                dpropname.get(),
                JS_GetStringBytes(
                  JS_ValueToString(context, propval)));
      continue;
    }

    JSObject* propobj = JSVAL_TO_OBJECT(propval);

    if(!search_only_prototypes) {

#if DEBUG_GN
      debug_gn_lvl += 4;
#endif

      const bool search_found_normal =
        SearchForJSVal(name, context, propobj,
                       search, visited_objects, 1);

#if DEBUG_GN
        debug_gn_lvl -= 4;
#endif

      if(search_found_normal) {
        found = FOUND_NORMAL;
        break;
      }
    }

    /* Since some objects have prototype as DontEnum, always visit the
     * prototype if the object has one and we haven't visited it
     * already. */
    jsval prototype = JSVAL_VOID;
    if(!JS_LookupProperty(context, propobj, "prototype", &prototype)) {
      DEBUG_GNP("JS_LookupProperty(\"prototype\") failed: %s",
                error_message(context).get());
      JS_ClearPendingException(context);
      continue;
    }

    if(JSVAL_IS_PRIMITIVE(prototype)) {
      DEBUG_GNP("prototype of \"%s\" is primitive: %s",
                dpropname.get(),
                JS_GetStringBytes(
                  JS_ValueToString(context, prototype)));
      continue;
    }

    JSObject* protoobj = JSVAL_TO_OBJECT(prototype);

    if(visited_objects.GetEntry((void*)protoobj)) {
      DEBUG_GNP("already visited proto of \"%s\"",
                dpropname.get());
      continue;
    }

    DEBUG_GNP("looking at proto of \"%s\"",
              dpropname.get());

    {
#if DEBUG_GN
      debug_gn_lvl += 4;
#endif

      const bool search_found_proto =
        SearchForJSVal(name, context, protoobj,
                       search, visited_objects, 1);

#if DEBUG_GN
      debug_gn_lvl -= 4;
#endif

      if(search_found_proto) {
        found = FOUND_PROTO;
        break;
      }
    }
  }

  if(found) {
    nsCString prefix = NS_ConvertUTF16toUTF8(
      (PRUnichar*)gent->name, gent->name_length);

    prefix += '.';

    name.Insert(prefix, 0);
    return true;
  }

  return false;
}


nsCString HRProfiler::GuessName(JSContext* cx, JSStackFrame* fp) {
  nsCString name;
  nsresult rv;
  JSFunction* func = JS_GetFrameFunction(cx, fp);
  JSString* own_name = func ? JS_GetFunctionId(func) : NULL;

  if(JS_IsNativeFrame(cx, fp)) {
    JSObject* thisp = JS_GetFrameThis(cx, fp);
    nsCString class_name;

    if(thisp) {
      nsCString js_class = hr_object_class_name(cx, thisp);

      // Try using XPCOM name if this is a native
      if(js_class.Find(NS_LITERAL_CSTRING("WrappedNative")) != -1) {
        nsCOMPtr<nsIXPConnectWrappedNative> wrapped_native;
        rv = mXPConnect->GetWrappedNativeOfJSObject(
          cx, thisp, getter_AddRefs(wrapped_native));

        if(NS_SUCCEEDED(rv)) {
          class_name += '[';

          nsCOMPtr<nsIInterfaceInfo> interface_info;
          const char* interface_name = NULL;

          if(own_name) {
            wrapped_native->FindInterfaceWithMember(
              STRING_TO_JSVAL(own_name),
              getter_AddRefs(interface_info));
            if(interface_info) {
              interface_info->GetNameShared(&interface_name);
            }
          }

          nsCOMPtr<nsIClassInfo> class_info =
            do_QueryWrappedNative(wrapped_native);

          nsCString contractid;
          if(class_info) {
            class_info->GetContractID(getter_Copies(contractid));
          }

          if(interface_name && contractid.Length()) {
            class_name += interface_name;
            class_name += NS_LITERAL_CSTRING(" of ");
            class_name += contractid;
          } else if(interface_info) {
            class_name += interface_name;
          } else if(contractid.Length()) {
            class_name += contractid;
          } else {
            class_name += NS_LITERAL_CSTRING("unknown native");
          }

          class_name += ']';
        }
      }

      // Otherwise just use the JS name
      if(!class_name.Length()) {
        class_name = js_class;
      }
    } else {
      // No thisptr?
      class_name = NS_LITERAL_CSTRING("Unknown");
    }

    name = class_name;
    name += NS_LITERAL_CSTRING("::");

    if(own_name) {
      name += JSString_to_nsCString(own_name);
    } else {
      name += NS_LITERAL_CSTRING("[constructor]");
    }
  } else {
    JSObject* scope = JS_GetScopeChain(cx);

    // Find global for object
    while(JS_GetParent(cx, scope)) {
      scope = JS_GetParent(cx, scope);
    }

    nsCString scopename = hr_object_class_name(cx, scope);
    bool found = false;
    bool do_search = true;

    if(own_name && mFlags & HR_PREFER_GIVEN_NAMES) {
      do_search = false;
    } else if(scopename.EqualsLiteral("ChromeWindow")) {
      do_search = mFlags & HR_SEARCH_CHROME_WINDOW;
    } else if(scopename.EqualsLiteral("Window")) {
      do_search = mFlags & HR_SEARCH_WINDOW;
    } else {
      do_search = mFlags & HR_SEARCH_OTHER;
    }

    if(do_search) {
      ptr_set visited_objects;
      JSObject* callee = JS_GetFrameCalleeObject(cx, fp);
      PR_ASSERT(callee);
      visited_objects.Init(1);

      DEBUG_GNP("\n---START- looking for %s %p scope=%s-------------",
                func ? JS_GetFunctionName(func) : "[native]",
                (void*)callee,
                scopename.get());

      found = SearchForJSVal(
        name, cx, scope,
        OBJECT_TO_JSVAL(callee),
        visited_objects,
        0);

      if(!found && scopename.EqualsLiteral("Window")) {
        found = SearchGlobalNames(name, cx,
                                  JS_GetGlobalObject(cx),
                                  OBJECT_TO_JSVAL(callee),
                                  visited_objects);
      }

      DEBUG_GNP("\n---END------------------\n\n");
    }

    if(!found) {
      if(own_name) {
        name = NS_LITERAL_CSTRING("[");
        name += JSString_to_nsCString(own_name);
        name += NS_LITERAL_CSTRING("]");
      } else {
        name = NS_LITERAL_CSTRING("[???]");
      }
    }
  }

  return name;
}

/**
 * Creates a hash key for script based on its name (if present),
 * filename, and line number. The key is a string allocated in the
 * thread-local buffer.
 */
inline static const char* make_script_hash_key(JSContext* cx,
                                               JSStackFrame* fp,
                                               JSScript* script,
                                               HRTLI* tli)
{
  PR_ASSERT(cx);
  PR_ASSERT(script);
  PR_ASSERT(tli);

  JSFunction* func = JS_GetFrameFunction(cx, fp);
  JSString* js_own_name = func ? JS_GetFunctionId(func) : NULL;
  const char* func_name = "";

  if(js_own_name) {
    func_name = JS_GetStringBytes(js_own_name);
  }

  size_t func_name_len = strlen(func_name);

  const char* filename = JS_GetScriptFilename(cx, script);
  if((JSUword)filename == JSFILENAME_NULL) {
    return NULL;
  }

  size_t filename_len = strlen(filename);

  char lineno_buf[30]; // More than maximum number of characters in
                       // the UINT64_MAX
  size_t lineno_buf_len = (size_t) sprintf(
    lineno_buf, "%lu",
    (unsigned long)JS_GetScriptBaseLineNumber(cx, script));

  size_t total_size = ( func_name_len  +  1 +
                        filename_len   +  1 +
                        lineno_buf_len +  1 );

  char* buf = (char*)tli->alloc_tmpbuf(total_size);

  if(likely(buf != NULL)) {
    char* pos = (char*)buf;
    memcpy(pos, func_name, func_name_len);
    pos += func_name_len;
    *(pos++) = '@';

    memcpy(pos, filename, filename_len);
    pos += filename_len;
    *(pos++) = ':';

    memcpy(pos, lineno_buf, lineno_buf_len);
    pos += lineno_buf_len;
    *(pos++) = 0;

    PR_ASSERT(pos - buf == (int)total_size);
  }

  return buf;
}

const HRFuncInfo* HRProfiler::GetFuncInfo(JSContext* cx,
                                          JSStackFrame* fp,
                                          bool* was_cached)
{
  HRFuncInfo* info = NULL;
  const bool do_lock = mFlags & HR_MAIN_THREAD_ONLY;
  JSObject* callee = JS_GetFrameCalleeObject(cx, fp);
  const char* script_hash_key = NULL;

  PR_ASSERT(callee);

  if(do_lock) {
    PR_Lock(mLock);
  }

  *was_cached = mFunctionCache.Get((void*)callee, &info);
  if(*was_cached) {
    if(do_lock) {
      PR_Unlock(mLock);
    }

    PR_ASSERT(info);
    return info;
  }

  JSScript* script = JS_GetFrameScript(cx, fp);
  if(script && (mFlags & HR_USE_FILELINE_CACHE)) {
    script_hash_key = make_script_hash_key(cx, fp, script, GetTLI());
    *was_cached = mFileLineCache.Get(script_hash_key, &info);
    if(*was_cached) {
      if(do_lock) {
        PR_Unlock(mLock);
      }

      PR_ASSERT(info);
      return info;
    }
  }

  info = new HRFuncInfo();
  info->name = GuessName(cx, fp);
  info->refcount = 1;

  if(script) {
    const char* script_filename = JS_GetScriptFilename(cx, script);
    if((JSUword)script_filename == JSFILENAME_NULL) {
      script_filename = "[null]";
    }

    info->filename = nsDependentCString(script_filename);
    info->lineno = JS_GetScriptBaseLineNumber(cx, script);
  } else {
    info->filename = NS_LITERAL_CSTRING("");
    info->lineno = 0;
  }

  if(script_hash_key) {
    info->refcount += 1;
    mFileLineCache.Put(script_hash_key, info);
  } else {
    mFunctionCache.Put((void*)callee, info);
  }

  if(do_lock) {
    PR_Unlock(mLock);
  }

  return info;
}

void HRProfiler::WriteEnt(HREnt* ent) {
  const bool do_lock = mFlags & HR_MAIN_THREAD_ONLY;
  if(do_lock) {
    PR_Lock(mLock);
  }

  PR_ASSERT(ent);
  PRUint32 to_write = ent->size;
  char* ptr = (char*)ent;

  if(!mOutput) {
    NS_WARNING("profile data race");
    goto out;
  }

  while(to_write) {
    PRUint32 written;
    if(NS_FAILED(mOutput->Write(ptr, to_write, &written))) {
      NS_WARNING("write failed!");
      goto out;
    }

    ptr += written;
    to_write -= written;
  }

out:
  if(do_lock) {
    PR_Unlock(mLock);
  }
}

static nsresult read_ent_header(nsIInputStream* input,
                                HREnt** entp,
                                size_t* alloc_size,
                                bool* at_eof)
{
  PR_ASSERT(input);
  PR_ASSERT(entp);
  PR_ASSERT(alloc_size);

  HREnt* ent = *entp;

  if(*alloc_size < sizeof(*ent)) {
    HREnt* tmp = (HREnt*)PR_Realloc(ent, sizeof(*ent));
    if(likely(tmp != NULL)) {
      *entp = ent = tmp;
      *alloc_size = sizeof(*ent);
    } else {
      return NS_ERROR_OUT_OF_MEMORY;
    }
  }

  PR_ASSERT(*alloc_size >= sizeof(*ent));
  return read_all(input, ent, sizeof(*ent), at_eof);
}

static nsresult read_ent_remainder(nsIInputStream* input,
                                   HREnt** entp,
                                   size_t* alloc_size)
{
  PR_ASSERT(input);
  PR_ASSERT(entp);
  PR_ASSERT(alloc_size);
  PR_ASSERT(*entp);

  HREnt* ent = *entp;

  if(*alloc_size < ent->size + 1) {
    HREnt* tmp = (HREnt*)PR_Realloc(ent, ent->size + 1);
    if(likely(tmp != NULL)) {
      *entp = ent = tmp;
      *alloc_size = ent->size + 1;
    } else {
      return NS_ERROR_OUT_OF_MEMORY;
    }
  }

  bool at_eof;
  nsresult rv = read_all(input, ((char*)ent) + sizeof(*ent),
                         ent->size - sizeof(*ent), &at_eof);
  NS_ENSURE_SUCCESS(rv, rv);
  if(at_eof) {
    NS_ERROR("truncated file");
    return NS_ERROR_FAILURE;
  }

  ((char*)ent)[ent->size] = 0; // ensure null-terminated even on corrupt input

  return NS_OK;
}

class HRProfilerEntry : public IHRProfilerEntry, public nsIClassInfo
{
  public:
  NS_DECL_ISUPPORTS
  NS_DECL_IHRPROFILERENTRY
  NS_DECL_NSICLASSINFO

  // Takes ownership of info ptr
  HRProfilerEntry(nsIInputStream* stream, HRInfoEnt* info, size_t alloc_size);

  private:
  ~HRProfilerEntry();

  nsCOMPtr<nsIInputStream> mInput;
  HRInfoEnt* mInfoEnt;
  HREnt* mEnt;
  size_t mEntAllocSize;

  static const unsigned int STATUS_READING = 0;
  static const unsigned int STATUS_AT_EOF = 1;
  static const unsigned int STATUS_ERROR = 2;
  unsigned int mStatus;
};

NS_IMPL_ISUPPORTS2(HRProfilerEntry, IHRProfilerEntry, nsIClassInfo)
NS_IMPL_CI_INTERFACE_GETTER1(HRProfilerEntry, IHRProfilerEntry)
IMPL_CI(HRProfilerEntry, "HRProfilerEntry", 0)

/* nsIVariant parseProfilerData (in nsIInputStream input); */
NS_IMETHODIMP HRProfiler::ParseProfilerData(nsIInputStream *input,
                                            IHRProfilerEntry** _retval)
{
  if(!input || !_retval) {
    return NS_ERROR_NULL_POINTER;
  }

  HREnt* ent = NULL;
  size_t alloc_size = 0;
  nsresult rv;
  bool at_eof;
  HRProfilerEntry* com_ent = NULL;

  rv = read_ent_header(input, &ent, &alloc_size, &at_eof);
  if(!NS_SUCCEEDED(rv)) {
    goto out;
  }

  if(at_eof) {
    NS_ERROR("empty file");
    rv = NS_ERROR_FAILURE;
    goto out;
  }

  if(ent->type != HRENT_TYPE_INFO) {
    NS_ERROR("file must start with info block");
    rv = NS_ERROR_FAILURE;
    goto out;
  }

  rv = read_ent_remainder(input, &ent, &alloc_size);
  if(!NS_SUCCEEDED(rv)) {
    goto out;
  }

  com_ent = new HRProfilerEntry(input, (HRInfoEnt*)ent, alloc_size);
  if(unlikely(!com_ent)) {
    rv = NS_ERROR_OUT_OF_MEMORY;
    goto out;
  }

  NS_ADDREF(com_ent);
  ent = NULL; // com_ent takes ownership of this

  rv = com_ent->Next();
  if(!NS_SUCCEEDED(rv)) {
    goto out;
  }

  *_retval = com_ent;
  com_ent = NULL; // don't release
  rv = NS_OK;

  out:
  NS_IF_RELEASE(com_ent);
  PR_Free(ent);
  return rv;
}

HRProfilerEntry::HRProfilerEntry(nsIInputStream* input,
                                 HRInfoEnt* info_ent,
                                 size_t alloc_size)
  : mInput(input),
    mInfoEnt((HRInfoEnt*)copy_ent((HREnt*)info_ent)),
    mEnt((HREnt*)info_ent),
    mEntAllocSize(alloc_size),
    mStatus(HRProfilerEntry::STATUS_READING)
{}

HRProfilerEntry::~HRProfilerEntry() {
  PR_Free(mInfoEnt);
  PR_Free(mEnt);
}

/* readonly attribute boolean searchedChromeWindow; */
NS_IMETHODIMP HRProfilerEntry::GetSearchedChromeWindow(PRBool *aSearchedChromeWindow)
{
  *aSearchedChromeWindow = mInfoEnt->flags & HR_SEARCH_CHROME_WINDOW;
  return NS_OK;
}

/* readonly attribute boolean searchedWindow; */
NS_IMETHODIMP HRProfilerEntry::GetSearchedWindow(PRBool *aSearchedWindow)
{
  *aSearchedWindow = mInfoEnt->flags & HR_SEARCH_WINDOW;
  return NS_OK;
}

/* readonly attribute boolean searchedOther; */
NS_IMETHODIMP HRProfilerEntry::GetSearchedOther(PRBool *aSearchedOther)
{
  *aSearchedOther = mInfoEnt->flags & HR_SEARCH_OTHER;
  return NS_OK;
}

/* readonly attribute boolean searchedOnlyPrototypes; */
NS_IMETHODIMP HRProfilerEntry::GetSearchedOnlyPrototypes(PRBool *aSearchedOnlyPrototypes)
{
  *aSearchedOnlyPrototypes = mInfoEnt->flags & HR_SEARCH_ONLY_PROTOTYPES;
  return NS_OK;
}

/* readonly attribute boolean preferredGivenNames; */
NS_IMETHODIMP HRProfilerEntry::GetPreferredGivenNames(PRBool *aPreferredGivenNames)
{
  *aPreferredGivenNames = mInfoEnt->flags & HR_PREFER_GIVEN_NAMES;
  return NS_OK;
}

NS_IMETHODIMP HRProfilerEntry::GetMaxDepth(PRUint16* aMaxDepth) {
  *aMaxDepth = mInfoEnt->maxDepth;
  return NS_OK;
}

NS_IMETHODIMP HRProfilerEntry::GetTimeSourceIsCounter(PRBool* aIsCounter) {
  *aIsCounter = (bool)mInfoEnt->timeSourceIsCounter;
  return NS_OK;
}

NS_IMETHODIMP HRProfilerEntry::GetMainThreadOnly(PRBool* aMainThreadOnly) {
  *aMainThreadOnly = mInfoEnt->flags & HR_MAIN_THREAD_ONLY;
  return NS_OK;
}

/* readonly attribute boolean usedFileLineCache; */
NS_IMETHODIMP HRProfilerEntry::GetUsedFileLineCache(PRBool *aUsedFileLineCache)
{
  *aUsedFileLineCache = mInfoEnt->flags & HR_USE_FILELINE_CACHE;
  return NS_OK;
}

/* readonly attribute string timeSource; */
NS_IMETHODIMP HRProfilerEntry::GetTimeSource(char * *aTimeSource)
{
  *aTimeSource = nsstrdup(mInfoEnt->clockname);
  return NS_OK;
}

NS_IMETHODIMP HRProfilerEntry::Next() {
  if(mStatus == HRProfilerEntry::STATUS_ERROR) {
    return NS_ERROR_FAILURE;
  }

  bool at_eof;
  nsresult rv = read_ent_header(mInput, &mEnt, &mEntAllocSize, &at_eof);
  NS_ENSURE_SUCCESS(rv, rv);

  if(at_eof) {
    mStatus = HRProfilerEntry::STATUS_AT_EOF;
    return NS_OK;
  }

  rv = read_ent_remainder(mInput, &mEnt, &mEntAllocSize);
  if(!NS_SUCCEEDED(rv)) {
    mStatus = HRProfilerEntry::STATUS_ERROR;
    return rv;
  }

  if(mEnt->type == HRENT_TYPE_INFO) {
    NS_WARNING("extra info entry; ignoring");
    return Next();
  }

  if(mEnt->type != HRENT_TYPE_CALL && mEnt->type != HRENT_TYPE_RETURN) {
    mStatus = HRProfilerEntry::STATUS_ERROR;
    NS_ERROR("unknown entry type");
    return NS_ERROR_FAILURE;
  }

  return rv;
}

/* readonly attribute boolean atEOF; */
NS_IMETHODIMP HRProfilerEntry::GetAtEOF(PRBool *aAtEOF)
{
  switch(mStatus) {
    case HRProfilerEntry::STATUS_READING:
      *aAtEOF = false;
      break;
    case HRProfilerEntry::STATUS_AT_EOF:
      *aAtEOF = true;
      break;
    default:
      return NS_ERROR_NOT_AVAILABLE;
  }

  return NS_OK;
}

/* readonly attribute unsigned long type; */
NS_IMETHODIMP HRProfilerEntry::GetType(PRUint32 *aType)
{
  if(mStatus != HRProfilerEntry::STATUS_READING) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  if(mEnt->type == HRENT_TYPE_CALL) {
    *aType = IHRProfilerEntry::TYPE_CALL;
  } else {
    PR_ASSERT(mEnt->type == HRENT_TYPE_RETURN);
    *aType = IHRProfilerEntry::TYPE_RETURN;
  }

  return NS_OK;
}

/* readonly attribute string name; */
NS_IMETHODIMP HRProfilerEntry::GetName(char * *aName)
{
  if(mStatus != HRProfilerEntry::STATUS_READING) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  if(mEnt->type == HRENT_TYPE_RETURN) {
    HRReturnEnt* rEnt = (HRReturnEnt*)mEnt;
    *aName = nsstrdup(rEnt->name_and_filename);
    if(unlikely(!*aName)) {
      return NS_ERROR_OUT_OF_MEMORY;
    }
  } else {
    *aName = NULL;
  }

  return NS_OK;
}

/* readonly attribute string filename; */
NS_IMETHODIMP HRProfilerEntry::GetFilename(char * *aFilename)
{
  if(mStatus != HRProfilerEntry::STATUS_READING) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  if(mEnt->type == HRENT_TYPE_RETURN) {
    HRReturnEnt* rEnt = (HRReturnEnt*)mEnt;
    size_t name_len = strlen(rEnt->name_and_filename);

    *aFilename = nsstrdup(rEnt->name_and_filename + name_len + 1);
    if(unlikely(!*aFilename)) {
      return NS_ERROR_OUT_OF_MEMORY;
    }
  } else {
    *aFilename = NULL;
  }

  return NS_OK;
}

/* readonly attribute unsigned short lineno; */
NS_IMETHODIMP HRProfilerEntry::GetLineno(PRUint32 *aLineno)
{
  if(mStatus != HRProfilerEntry::STATUS_READING) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  if(mEnt->type == HRENT_TYPE_RETURN) {
    HRReturnEnt* rEnt = (HRReturnEnt*)mEnt;
    *aLineno = rEnt->lineno;
  } else {
    *aLineno = 0;
  }

  return NS_OK;
}

/* readonly attribute unsigned long long startValue; */
NS_IMETHODIMP HRProfilerEntry::GetStartValue(PRUint64 *aStartValue)
{
  if(mStatus != HRProfilerEntry::STATUS_READING) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  if(mEnt->type == HRENT_TYPE_RETURN) {
    HRReturnEnt* rEnt = (HRReturnEnt*)mEnt;
    *aStartValue = rEnt->nanoseconds_start;
  } else {
    PR_ASSERT(mEnt->type == HRENT_TYPE_CALL);
    HRCallEnt* cEnt = (HRCallEnt*)mEnt;
    *aStartValue = cEnt->nanoseconds;
  }

  return NS_OK;
}

/* readonly attribute unsigned long long endValue; */
NS_IMETHODIMP HRProfilerEntry::GetEndValue(PRUint64 *aEndValue)
{
  if(mStatus != HRProfilerEntry::STATUS_READING) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  if(mEnt->type == HRENT_TYPE_RETURN) {
    HRReturnEnt* rEnt = (HRReturnEnt*)mEnt;
    *aEndValue = rEnt->nanoseconds_end;
  } else {
    PR_ASSERT(mEnt->type == HRENT_TYPE_CALL);
    HRCallEnt* cEnt = (HRCallEnt*)mEnt;
    *aEndValue = cEnt->nanoseconds;
  }

  return NS_OK;
}

/* readonly attribute unsigned long thread */
NS_IMETHODIMP HRProfilerEntry::GetThread(PRUint32 *aThread) {
  if(mStatus != HRProfilerEntry::STATUS_READING) {
    return NS_ERROR_NOT_AVAILABLE;
  }

  if(mEnt->type == HRENT_TYPE_RETURN) {
    HRReturnEnt* rEnt = (HRReturnEnt*)mEnt;
    *aThread = rEnt->thread;
  } else {
    PR_ASSERT(mEnt->type == HRENT_TYPE_CALL);
    HRCallEnt* cEnt = (HRCallEnt*)mEnt;
    *aThread = cEnt->thread;
  }

  return NS_OK;
}

/* readonly attribute unsigned long callee */
NS_IMETHODIMP HRProfilerEntry::GetCallee(PRUint32 *aCallee) {
  if(mStatus != HRProfilerEntry::STATUS_READING) {
    return NS_ERROR_NOT_AVAILABLE;
  }

#if RECORD_CALLEE_IN_LOGS

  if(mEnt->type == HRENT_TYPE_RETURN) {
    HRReturnEnt* rEnt = (HRReturnEnt*)mEnt;
    *aCallee = rEnt->callee;
  } else {
    PR_ASSERT(mEnt->type == HRENT_TYPE_CALL);
    HRCallEnt* cEnt = (HRCallEnt*)mEnt;
    *aCallee = cEnt->callee;
  }

  return NS_OK;

#else
  return NS_ERROR_NOT_AVAILABLE;
#endif

}

static nsCString error_message(JSContext* context) {
  jsval exception = JSVAL_NULL;
  JS_GetPendingException(context, &exception);
  JSString* str = JS_ValueToString(context, exception);
  return nsCString(str ? JS_GetStringBytes(str) : "?");
}

static nsresult read_all(nsIInputStream* input,
                         void* buf_in,
                         size_t bytes,
                         bool* at_eof)
{
  char* buf = (char*)buf_in;
  nsresult rv;
  *at_eof = false;

  while(bytes) {
    PRUint32 bytes_read;
    rv = input->Read(buf, bytes, &bytes_read);

    if(!bytes_read) {
      *at_eof = true;
      return NS_OK;
    }

    NS_ENSURE_SUCCESS(rv, rv);

    buf += bytes_read;
    bytes -= bytes_read;
  }

  return NS_OK;
}

