/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is MacroTracker Bug Tracking System Source Code
 *
 * The Initial Developer of the Original Code is
 * R.J. Keller.
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */
package org.mozdev.MacroTracker.toolkit.guiPanels;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import org.mozdev.MacroTracker.classes.*;
import org.mozdev.MacroTracker.toolkit.*;

import com.trfenv.rlk.Debug;

/**
 *Displays the user editor panel so the user can edit users. This is 
 *convient when you would like to make the user editor panel in its own
 *window, or inside of another UI element. The user list is generated from
 *the master bug database in class MacroTracker.
 *
 *<p>Any modifications made to users will not be made directly to the bug
 *  database. To make those modifications, you can use a iterator returned
 *  from the getUsers function to add and remove users approprately
 *  depending on modifications.</p>
 *
 *@author R.J. Keller
 */
public class UserEditorPanel extends JPanel implements ActionListener
{
	private MComboBox cboUsers;
	private MTextField txtName;
	private MTextField txtEmail;
	private MTextField txtPassword;
	
	private JCheckBox chkCanConfirm;
	private JCheckBox chkEditBugs;
	private JCheckBox chkEditComponents;
	private JCheckBox chkEditKeywords;
	private JCheckBox chkEditUsers;
	
	private JButton btnAdd;
	private JButton btnApply;
	private JButton btnHelp;
	private JButton btnRemove;
	
	private LinkedList mUsers;

	/**
	 *Creates a new UserEditorPanel with the users from the bug database
	 *already taken in set in the GUI for editing.
	 */
	public UserEditorPanel(iBugDatabase database)
	{
		this();
		for (Iterator i = database.userIterator(); i.hasNext();)
		{
			User j = (User)i.next();
			cboUsers.addItem(j.getEmail());
			mUsers.add(j);
		}
	}

	/**
	 *Creates a standard UserEditorPanel with no users added to the list
	 *initially.
	 */
	public UserEditorPanel()
	{
		setLayout(new BorderLayout());
		mUsers = new LinkedList();
		
		//adding cboUsers and btnEdit
		JPanel pnlUsers = new JPanel(new BorderLayout());
		cboUsers = new MComboBox();
		pnlUsers.add(cboUsers, BorderLayout.CENTER);
		
		btnRemove = new JButton("Remove");
		JPanel pnlFlowLayout = new JPanel(new FlowLayout());
		pnlFlowLayout.add(btnRemove);
		pnlUsers.add(pnlFlowLayout, BorderLayout.EAST);
		
		add(pnlUsers, BorderLayout.NORTH);
		
		
		JPanel pnlMainContent = new JPanel(new GridLayout(8,1));
		
		//adding txtName
		JPanel pnlName = new JPanel(new BorderLayout());
		txtName = new MTextField();
		pnlName.add(new JLabel("Name:", JLabel.RIGHT), BorderLayout.WEST);
		pnlName.add(txtName, BorderLayout.CENTER);
		pnlMainContent.add(pnlName);
		
		//adding txtEmail
		JPanel pnlEmail = new JPanel(new BorderLayout());
		txtEmail = new MTextField();
		pnlEmail.add(new JLabel("Email:", JLabel.RIGHT), BorderLayout.WEST);
		pnlEmail.add(txtEmail, BorderLayout.CENTER);
		pnlMainContent.add(pnlEmail);
		
		//adding txtPassword
		JPanel pnlPassword = new JPanel(new BorderLayout());
		txtPassword = new MTextField();
		pnlPassword.add(new JLabel("Password:", JLabel.RIGHT), BorderLayout.WEST);
		pnlPassword.add(txtPassword, BorderLayout.CENTER);
		pnlMainContent.add(pnlPassword);
		
		//adding check boxes.
		chkCanConfirm = new JCheckBox("Can confirm bugs.");
		chkEditBugs = new JCheckBox("Can edit all bugs.");
		chkEditComponents = new JCheckBox("Can modify products and components.");
		chkEditKeywords = new JCheckBox("Can modify keywords.");
		chkEditUsers = new JCheckBox("Can modify user privileges and properties.");
		
		pnlMainContent.add(chkCanConfirm);
		pnlMainContent.add(chkEditBugs);
		pnlMainContent.add(chkEditComponents);
		pnlMainContent.add(chkEditKeywords);
		pnlMainContent.add(chkEditUsers);
		
		add(pnlMainContent, BorderLayout.CENTER);
		
		//adding btnAdd, btnApply, btnHelp
		btnAdd = new JButton("Add");
		btnApply = new JButton("Apply");
		btnHelp = new JButton("Help");
		
		JPanel pnlButtons = new JPanel(new FlowLayout());
		pnlButtons.add(btnAdd);
		pnlButtons.add(btnApply);
		pnlButtons.add(btnHelp);
		
		add(pnlButtons, BorderLayout.SOUTH);
		
		//setting action listeners
		btnAdd.addActionListener(this);
		btnApply.addActionListener(this);
		btnRemove.addActionListener(this);
		cboUsers.addActionListener(this);
	}
	
	/**
	 *Returns an iterator that iterates through all of the users in the list,
	 *including those that are added and removed.
	 */
	public Iterator getUsers()
	{
		return mUsers.iterator();
	}
	
	public void actionPerformed(ActionEvent e)
	{
		if (e.getSource() == btnAdd)
		{
			Debug.println("textbox.text() == \"" + txtEmail.getText() + "\";");
			if (txtEmail.getText().equals(""))
			{
				JOptionPane.showMessageDialog(null, "In order to add a user, you need at least their email address filled in.");
				return;
			}
			User newUser = new User(txtName.getText(), txtEmail.getText());
			Debug.println("txtName.getText() == " + txtName.getText());
			Debug.println("txtEmail.getText() == " + txtEmail.getText());
			newUser.setCanConfirm(chkCanConfirm.isSelected());
			newUser.setEditBugs(chkEditBugs.isSelected());
			newUser.setEditComponents(chkEditComponents.isSelected());
			newUser.setEditKeywords(chkEditKeywords.isSelected());
			newUser.setEditUsers(chkEditUsers.isSelected());
			
			mUsers.add(newUser);
			cboUsers.addItem(newUser.getEmail());
		}
		else if (e.getSource() == btnApply)
		{
			try
			{
				User getUser = (User)mUsers.get(cboUsers.getSelectedIndex());
				getUser.setCanConfirm(chkCanConfirm.isSelected());
				getUser.setEditBugs(chkEditBugs.isSelected());
				getUser.setEditComponents(chkEditComponents.isSelected());
				getUser.setEditKeywords(chkEditKeywords.isSelected());
				getUser.setEditUsers(chkEditUsers.isSelected());
			}
			catch (Exception ex)
			{
				Debug.printException(ex);
			}
		}
		else if (e.getSource() == btnRemove)
		{
			try
			{
			int selectedIndex = cboUsers.getSelectedIndex();
			mUsers.remove(selectedIndex);
			}
			catch (Exception ex)
			{
				//an exception is thrown if there are no users in the list or if there
				//is not a selected index, so do nothing when this happens.
			}
			cboUsers.removeSelectedItem();
		}
		else if (e.getSource() == cboUsers)
		{
			User selUser = (User)mUsers.get(cboUsers.getSelectedIndex());
			txtName.setText(selUser.getName());
			txtEmail.setText(selUser.getEmail());
			txtPassword.setText(selUser.getPassword());
			
			chkCanConfirm.setSelected(selUser.canConfirm());
			chkEditBugs.setSelected(selUser.canEditBugs());
			chkEditComponents.setSelected(selUser.getEditComponents());
			chkEditKeywords.setSelected(selUser.canEditKeywords());
			chkEditUsers.setSelected(selUser.canEditUsers());
		}
	}
}
