/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla Help Viewer API in Java.
 *
 * The Initial Developer of the Original Code is
 * R.J. Keller.
 * Portions created by the Initial Developer are Copyright (C) 2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package org.mozdev.firebirdhelp.general;

import javax.swing.*;

import com.trfenv.rlk.Debug;

/**
 *This ToolbarButton class is a class that inherits JButton and simulates
 *functions that would be useful for a button on a JToolBar.
 *
 *<p>Functions added include the ability to swap between Images and Text,
 *Images only, or Text only and the ability to change between large and small
 *images to show in the button.</p>
 *
 *@author R.J. Keller <rlk@trfenv.com>
 */
public class ToolbarButton extends JButton
{
	private ImageIcon mLargeImage;
	private ImageIcon mSmallImage;
	private String mButtonTitle;
	
	private byte mButtonMode;
	private byte mPictureMode;
	
	public static final byte LARGE_PICTURE = 0;
	public static final byte SMALL_PICTURE = 1;
	
	public static final byte IMAGE_AND_TEXT = 0;
	public static final byte IMAGE_ONLY = 1;
	public static final byte TEXT_ONLY = 2;
	
	/**
	 *Creates a new ToolbarButton with its text already set.
	 *
	 *<p>The initialized modes set for ToolbarButton's when they are created
	 *  are IMAGE_AND_TEXT for the toolbar mode and SMALL_PICTURE for the
	 *  toolbar image.</p>
	 *@param str The text to make the ToolbarButton.
	 */
	public ToolbarButton(String str)
	{
		super();
		setText(str);
		
		setButtonMode(TEXT_ONLY);
	}
	
	/**
	 *Creates a new Toolbarbutton with the button's text and images already
	 *set.
	 *
	 *@param str The text to make the ToolbarButton.
	 *@param smallImgUrl The small image uri if the ToolbarButton image in the
	 *  image mode <code>SMALL_PICTURE</code>, which is the default toolbar
	 *  image mode.
	 *@param largeImgUrl The large image uri to display if the ToolbarButton
	 *  image mode is set to <code>LARGE_PICTURE</code>.
	 */
	public ToolbarButton(String str,
						String smallImgUrl,
						String largeImgUrl)
	{
		this(str);

		setSmallImg(smallImgUrl);
		setLargeImg(largeImgUrl);
		
		setButtonMode(IMAGE_AND_TEXT);
		setPictureMode(SMALL_PICTURE);
	}
	
	/**
	 *Sets whether or not the toolbar button will display a large or a small
	 *image.<br>
	 *(precondition: imageMode == LARGE_PICTURE, SMALL_PICTURE)
	 *@param imageMode Whether or not you want small images or large image to
	 *  display in this ToolbarButton
	 *@throws IllegalArgumentException if you did not take in a large or a small
	 *  image, but some other invalid value that cannot be parsed.
	 */
	public void setPictureMode(byte imageMode) throws IllegalArgumentException
	{
		if (mButtonMode == TEXT_ONLY)
			return;

		if (imageMode == LARGE_PICTURE)
		{
			Debug.println(mButtonTitle + ".imageMode => LARGE_PICTURE");
			super.setIcon(mLargeImage);
		}
		else if (imageMode == SMALL_PICTURE)
		{
			Debug.println(mButtonTitle + ".imageMode => SMALL_PICTURE");
			super.setIcon(mSmallImage);
		}
		else
		{
			throw new IllegalArgumentException("Invalid picture mode given for toolbar button.");
		}
		
		mPictureMode = imageMode;
	}
	
	/**
	 *Sets whether or not the button will display Images and Text, Text only,
	 *or Pictures only.<br>
	 *(precondition: buttonMode == IMAGE_AND_TEXT, IMAGE_ONLY, TEXT_ONLY)
	 *@param buttonMode How you want this ToolbarButton to be displayed on the
	 *  screen.
	 */
	public void setButtonMode(byte buttonMode) throws IllegalArgumentException
	{
		if (buttonMode == IMAGE_AND_TEXT)
		{
			Debug.println(mButtonTitle + ".buttonMode => IMAGE_AND_TEXT");

			setPictureMode(mPictureMode);
		}
		else if (buttonMode == IMAGE_ONLY)
		{
			Debug.println(mButtonTitle + ".buttonMode => IMAGE_ONLY");

			setPictureMode(mPictureMode);
		}
		else if (buttonMode == TEXT_ONLY)
		{
			Debug.println(mButtonTitle + ".buttonMode => TEXT_ONLY");

			super.setIcon(null);
		}
		else
		{
			throw new IllegalArgumentException("Invalid button mode given for toolbar button.");
		}
		
		setText(mButtonTitle);
	}
	
	/**
	 *Sets the uri of the toolbar image to display when the image mode is
	 *set to LARGE_PICTURE.
	 *
	 *<p>The toolbar image cannot be modified when the toolbar mode is set
	 *to TEXT_ONLY.</p>
	 *@param url The uri of the image you want to use as the large image.
	 */
	public void setLargeImg(String url)
	{
		Debug.println(mButtonTitle + "'s large image == " + url);
		try
		{
			mLargeImage = new ImageIcon(ToolbarButton.class.getResource(url), "");

			if (mButtonMode != TEXT_ONLY &&
				mPictureMode == LARGE_PICTURE)
				super.setIcon(mLargeImage);
		}
		catch (Exception ex)
		{
			Debug.printException(ex);
		}
	}
	
	/**
	 *Sets the image to display when the image mode is SMALL_PICTURE.
	 *@param url The uri of the image to display.
	 */
	public void setSmallImg(String url)
	{
		Debug.println(mButtonTitle + "'s small image == " + url);
		try
		{
			mSmallImage = new ImageIcon(ToolbarButton.class.getResource(url), "");

			if (mButtonMode != TEXT_ONLY &&
				mPictureMode == SMALL_PICTURE)
				super.setIcon(mSmallImage);

		}
		catch (Exception ex)
		{
			Debug.printException(ex);
		}		
	}
	
	/**
	 *Sets the text to display in the ToolbarButton. The text will not be
	 *displayed if the image mode is TEXT_ONLY.
	 *@param text The text to display.
	 */
	public void setText(String text)
	{
		mButtonTitle = text;

		if (mButtonMode != IMAGE_ONLY)
			super.setText(mButtonTitle);
		else
			super.setText("");

		super.setText("");
		Debug.println("toolbarButton->setText " + super.getText());
	}
	
	/**
	 *Returns the text that is displayed in the ToolbarButton. No text will be
	 *displayed if the toolbar mode is TEXT_ONLY, but the current set text
	 *will be returned.
	 */
	public String getText()
	{
		return mButtonTitle;
	}
}