"""Implements SHA-256 (see FIPS 180-2: Secure Hash Standard)."""

def rotate(n, shift):
    return (n>>shift) | (n<<(32-shift)) & 0xffffffff;

def choice(x, y, z):
    return (x & y) ^ (~x & z)

def majority(x, y, z):
    return (x & y) ^ (x & z) ^ (y & z)

def Sigma0(x):
    return rotate(x, 2) ^ rotate(x, 13) ^ rotate(x, 22)

def Sigma1(x):
    return rotate(x, 6) ^ rotate(x, 11) ^ rotate(x, 25)

def sigma0(x):
    return rotate(x, 7) ^ rotate(x, 18) ^ (x>>3)

def sigma1(x):
    return rotate(x, 17) ^ rotate(x, 19) ^ (x>>10)

def byte(n):
    return n % (1<<8)

def word(n):
    return n % (1<<32)

def ch(n):
    return chr(byte(n))

def toword(s):
    return (ord(s[0])<<24) | (ord(s[1])<<16) | (ord(s[2])<<8) | ord(s[3])

def tochars(w):
    return ch(w>>24) + ch(w>>16) + ch(w>>8) + ch(w)

K = [0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5,
     0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
     0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3,
     0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
     0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc,
     0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
     0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7,
     0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
     0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13,
     0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
     0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3,
     0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
     0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5,
     0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
     0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208,
     0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2]

def pad(message):
    l = len(message)*8
    message += '\x80'
    message += '\x00' * ((-len(message)-8) % 64)
    message += tochars(l>>32) + tochars(l)
    return message

def hash(message):
    message = pad(message)
    H = [0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a,
         0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19]
    for i in range(0, len(message), 64):
        block = message[i:i + 64]
        words = [toword(block[j:j + 4]) for j in range(0, 64, 4)]
        for t in range(16, 64):
            words.append(word(sigma1(words[t - 2]) + words[t - 7] +
                              sigma0(words[t - 15]) + words[t - 16]))
        a, b, c, d, e, f, g, h = H
        for t in range(64):
            T1 = h + Sigma1(e) + choice(e, f, g) + K[t] + words[t]
            T2 = Sigma0(a) + majority(a, b, c)
            h, g, f, e, d, c, b, a = \
                g, f, e, word(d + T1), c, b, a, word(T1 + T2)
        H = [word(x + y) for x, y in zip(H, [a, b, c, d, e, f, g, h])]
    return ''.join(map(tochars, H))
